#pragma once
#include <memory>   // for auto_ptr

namespace kofax {
namespace tna {



/**
  * base class for all objects returned by the interface via auto_ptr.
	*/
class CommonBase {
public:
	virtual ~CommonBase()=0;
};


/**
  * this class wraps the result of most function calls.
	*/
class ResultBase : public CommonBase {
public:
	/**
	  * returns true if the result indicates that the last operation was successful.
		* It is the same as (GetErrorCode() == 0).
		*/
	virtual bool        IsOK() const = 0;

	/**
	  * returns an error number where 0 indicates that the operation was successful.
		*/
	virtual int         GetErrorCode() const = 0;

	/**
	  * returns an error description as c-string. The string is valid until Result object is changed.
		*/
	virtual const char *GetErrorDescription() const = 0;
};

typedef std::auto_ptr<ResultBase> ResultPtr;

/**
  * Common base class for all interfaces.
	*/
class CommonInterface {
public:
	/**
	  * Initialize the interface with a configuration.
		*
		* @param sz_inst_path       path to installation directory of TNA_FAX. A relative path refers to the current working directory 
		*                           while this function was called.
		* @param sz_config          string with XML configuration for the interface
		* @param sz_xslt_file       optionally pathname to XSLT file which converts sz_config into internal configuration structure. A relative
		*                           path refers to sz_inst_path.
		* @param sz_int_config_file optionally pathname of a file where the actuall used configuration (after transformation) is stored. 
		*                           The file may be used for debugging only.  A relative path refers to sz_inst_path.
		*/
	virtual ResultPtr Start(const char *sz_inst_path, const char *sz_config, const char *sz_xslt_file=0, const char *sz_int_config_file=0)=0;

	/**
	  * Shutdown the interface. The function blocks until all threads used by the interface have been terminated and all resources
		* are freed. This function must be called before unloading the interface DLL. If ResultPtr.IsOK() is false, the shutdown operation
		* could not be completed within the specified time-out. In such a case resources used by the DLL may not be released and DLL unload 
		* may cause cause a crash.
		*
		* @param time_out   time-out until all resources have to be released in milliseconds.
		*/
	virtual ResultPtr Shutdown(int time_out=10000)=0;
};


} /* end of namespace tna */
} /* end of namespace kofax */