/**
  * This file describes the C++ API for fax outbound integration. 
	*
	* It is a sub-set of the existing web services used to integrate
	* fax with Kofax Communication Server, RightFax, BISCOM and KCS FoIP which
	* is provided as C++ interface instead of web-services.
	*
	* Details about the used values and compatibility with fax servers can be found 
	* in the documents: "Web Services for Fax Technical Manual" and "Fax Connector SDK"
	*
	* API History:
	* date        API(TNA_Fax)  Info
	* 2009-03-01  1000(1.00.02) First version released as part of TNC_Fax 1.00.02
	* 2011-01-27  1010(3.02.00) SendFaxFromFile2() with more options added
	*/

#pragma once
#include <tna_common.h>

namespace kofax {
namespace tna {

/**
  * This class encapsulates the response to the GetServerProperties call.
	*/
class ServerPropertiesBase : public CommonBase {
public:
	/**
	  * returns the fax server type as c-string. If the information is not available an empty string ("") is returned.
		* the string is valid until GetServerPropertyResponse is changed.
		*/
	virtual const char *GetServerType() const = 0;

	/**
	  * returns the version info of the used fax server as c-string. 
		* If the information is not available an empty string ("") is returned.
		* the string is valid until GetServerPropertyResponse is changed.
		*/
	virtual const char *GetFaxServerVersion() const = 0;
};
typedef std::auto_ptr<ServerPropertiesBase> ServerPropertiesPtr;

/**
  * This call defines the function calls on the Fax interface.
	*/
class FaxInterface : virtual public CommonInterface {
public:

	/**
	  * Defines the cover-sheet mode
	  */	
	enum CoverModes {
		COVER_NEVER,         //!< Send the Fax without cover-sheet
		COVER_SERVER_DEP,    //!< Cover-sheet enable/disable depends on Server/User-Profile (=default)
		COVER_ALWAYS         //!< Send the Fax with a cover-sheet
	}; 
	
	/**
	  * Defines the delivery security mode
	  */	
	enum DeliveryModes {
		DM_FAX=0,                //!< Send the messages as Fax (without Secure Document or Certified Delivery option)
		DM_SECURE_DOC=1,         //!< send the message as secure document (email with encrypted/signed PDF)
		DM_CERTIFIED_DELIVERY=3  //!< send the mesages as secure document with certified delivery
	}; 
	
	
	/**
	  * This constant defines the current version of the API. It will be incremented whenever the functionality of the API 
		* will be changed.
		*
		* @remark It is the version of the header-file used during compilation of the application. You must
		*         use function IsCompatible() in order to get the api-version of the DLL! 
		*/
	static const int API_VERSION=1010;

	/**
	  * This function returns the exported name of this interface within the DLL
		*/
	static const char *GetInterfaceName()
	{
		return "?GLOBAL_FaxInterface@tna@kofax@@3VFaxInterfaceImpl@12@A";
	}

	/**
	  * This function should be used to check the compatibility of the DLL-API with the used application. You may
		* either specific a specific version of the API (=expected_api_version) or a version range (
		*
		* @param expected_api_version  This parameter must be set to the API version used during build of the application. 
		*                          The fixed value FaxInterface::API_VERSION should be used.
		* @param min_api_version   You may optionaly specify the used API features using to lowest API which supports
		*                          all this features. If you do not know this version use 0.
		* @param p_current_api_version  If this parameter is != 0, the current API version of the DLL is returned. 
		* @retval true             The API is compatible with the application.
		* @retval false            The API is not compatible with the application.
		*/
	virtual bool IsCompatible(int expected_api_version, int min_api_version=0, int *p_current_api_version=0)=0;


	/**
	  * Get the actual fax server properties. 
		* It may also be used to check if the connection to the fax server works.
		* If sz_user is specified both sz_user and sz_password are used for user authentication.
		*
		* @remark if any string parameter is empty ("") or a 0-pointer, this parameter is handled as none specified. 
		*
		* @param p_response  receive ServerProperties response.
		* @param sz_user     login user Id. if sz_user == 0 or sz_user is "", no authentication information 
		* @param sz_password login password
		*/
	virtual ResultPtr GetServerProperties(ServerPropertiesPtr *p_response, const char *sz_user=0, const char *sz_password=0)=0;


	/**
	  * Send fax message from a TIFF file. This function provides a sub-set of SendFaxFromFile2() it is provided for 
	  * backward compatibility with applications build with API version 1000. See SendFaxFromFile2() for details.
	  */
	virtual ResultPtr SendFaxFromFile(
		const char    *sz_user, 
		const char    *sz_password,
		const char    *sz_tiff_file,
		const char    *sz_number_to_dial,
		const char    *sz_expected_csi=0,
		const char    *sz_name_of_recipient=0,
		const char    *sz_source_tsi=0,
		const char    *sz_name_of_sender=0,
		const char    *sz_subject=0
	)=0;


	/**
	  * Send fax message from a TIFF file.
		* If sz_user is specified both sz_user and sz_password are used for user authentication.
		*
		* @remark if any string parameter is empty ("") or a 0-pointer, this parameter is handled as none specified. 
		*
		* @param sz_user           login user Id
		* @param sz_password       login password
		* @param sz_tiff_file      pathname of TIFF file which should be sent
		* @param sz_number_to_dial destination fax number
		* @param sz_expected_csi   Expected called subscriber identification, or a part of it, for verification that the correct 
		*                          destination has been called. The sending proceeds only if the string specified here can be found 
		*                          somewhere within the received CSI.
		* @param sz_name_of_recipient Recipient name inserted into fax header line
		* @param sz_source_tsi     Originating subscriber identification, the recipient should send a reply to this fax number.
		* @param sz_name_of_sender Sender name inserted into fax header line
		* @param sz_subject        The message's subject field. It is not inserted into the transmitted fax, but visible on the fax server,
		*                          may be useful for auditing and trouble-shooting.
		* @param cover_mode        Defines if the fax should be sent with a Cover-sheet. See enum CoverModes
		* @param sz_owner_id       Specifies the originator if the login user sents the fax on behalf of OwnerId.
		* @param sz_bill_code1     Billing code 1
		* @param sz_bill_code2     Billing code 2
		* @param sec_delivery_mode Enables Secure-Document / Certified-Delivery. See enum DeliveryModes
		* @param sz_sec_delivery_password  Optional password for Secure-Document / Certified-Delivery
		*/
	virtual ResultPtr SendFaxFromFile2(
		const char    *sz_user, 
		const char    *sz_password,
		const char    *sz_tiff_file,
		const char    *sz_number_to_dial,
		const char    *sz_expected_csi=0,
		const char    *sz_name_of_recipient=0,
		const char    *sz_source_tsi=0,
		const char    *sz_name_of_sender=0,
		const char    *sz_subject=0,
		CoverModes    cover_mode=COVER_SERVER_DEP,
		const char    *sz_owner_id=0,
		const char    *sz_bill_code1=0,
		const char    *sz_bill_code2=0,
		DeliveryModes delivery_mode=DM_FAX,
		const char    *sz_sec_delivery_password=0
	)=0;
};


} /* end of namespace tna */
} /* end of namespace kofax */