'****************************************************************************
'  (c) Copyright 2009 Kofax, Inc.. All rights reserved.
'  Unauthorized use, duplication or distribution is strictly prohibited.
'****************************************************************************
'
' File: ActionEventHandler.vb
'
' Purpose: Defines the class for handling KC events.
'
' ****************************************************************************

Imports Kofax.TAPCommon.TAPLib
Imports Kofax.ReleaseLib

''' <summary>
''' Defines the class for handling KC events.
''' </summary>
''' <remarks></remarks>
Public Class ActionEventHandler

#Region "Member declaration"
	Private m_oSetupDataNet				As IReleaseSetupDataNet
    Private m_bShowUI					As Boolean
    Private m_bSuppressUI As Boolean
    Private m_strDeletedDestinationID As String
    Private m_bDirty As Boolean
#End region

#Region "Properties"
    Public ReadOnly Property IsDirty() As Boolean
        Get
            Return m_bDirty
        End Get
    End Property

    Public ReadOnly Property DeletedDestinationID() As String
        Get
            Return m_strDeletedDestinationID
        End Get
    End Property
#End Region

#Region "Public funtions"

	''' <summary>
	''' Implements IKfxReleaseSetupScript.ActionEvent 
	''' </summary>
	''' <param name="oSetupDataNet"></param>
	''' <param name="ActionID"></param>
	''' <param name="strData1"></param>
	''' <param name="strData2"></param>
	''' <returns></returns>
	''' <remarks></remarks>
	Public Function ActionEvent(ByVal oSetupDataNet As IReleaseSetupDataNet, _
			ByVal ActionID As KfxActionValue, _
			ByRef strData1 As String, _
			ByRef strData2 As String) As KfxReturnValue

		Dim eReturnValue As KfxReturnValue = KfxReturnValue.KFX_REL_SUCCESS

		If oSetupDataNet Is Nothing Then
			Throw New ArgumentException("Null ReleaseSetupData in ActionEvent", "oSetupData")
		End If

		m_oSetupDataNet = oSetupDataNet

		Try

			Select Case ActionID
				Case KfxActionValue.KFX_REL_START
                    ' Initializes to false.  Certain action events will set this value as needed.
					m_bShowUI = False
                    m_bDirty = False

				Case KfxActionValue.KFX_REL_BATCHFIELD_DELETE
					OnFieldDelete(strData1, KfxLinkSourceType.KFX_REL_BATCHFIELD)

				Case KfxActionValue.KFX_REL_BATCHFIELD_RENAME
					OnFieldRename(strData1, strData2, KfxLinkSourceType.KFX_REL_BATCHFIELD)

				Case KfxActionValue.KFX_REL_BATCHFIELD_INSERT
                    ' SPR00049027 - KCEC-Fax: Not all Action Events are handled
                    OnFieldInsert(strData1, KfxLinkSourceType.KFX_REL_BATCHFIELD)

				Case KfxActionValue.KFX_REL_INDEXFIELD_DELETE
					OnFieldDelete(strData1, KfxLinkSourceType.KFX_REL_INDEXFIELD)

				Case KfxActionValue.KFX_REL_INDEXFIELD_RENAME
					OnFieldRename(strData1, strData2, KfxLinkSourceType.KFX_REL_INDEXFIELD)

				Case KfxActionValue.KFX_REL_INDEXFIELD_INSERT
                    ' SPR00049027 - KCEC-Fax: Not all Action Events are handled
                    OnFieldInsert(strData1, KfxLinkSourceType.KFX_REL_INDEXFIELD)
                Case KfxActionValue.KFX_REL_FOLDERCLASS_INSERT
                    ' SPR00049027 - KCEC-Fax: Not all Action Events are handled
                    OnFieldInsert(strData1, KfxLinkSourceType.KFX_REL_INDEXFIELD)
                Case KfxActionValue.KFX_REL_FOLDERCLASS_DELETE
                    OnFieldDelete(strData1, KfxLinkSourceType.KFX_REL_INDEXFIELD)

                Case KfxActionValue.KFX_REL_FOLDERCLASS_RENAME
                    OnFieldRename(strData1, strData2, KfxLinkSourceType.KFX_REL_INDEXFIELD)
                Case KfxActionValue.KFX_REL_PUBLISH_CHECK
                    eReturnValue = OnPublishCheck()

				Case KfxActionValue.KFX_REL_UPGRADE
					eReturnValue = OnUpgrade()

				Case KfxActionValue.KFX_REL_END
					' Saves any action event changes
					m_oSetupDataNet.Apply()    
					' Runs the UI if flag was set by an action event
					If m_bShowUI Then
						m_bShowUI = False  
						Dim oSetup As New Setup
						If Not m_bSuppressUI Then
                            eReturnValue = oSetup.ShowForm(m_oSetupDataNet, Me)
                        End If
					End If
				' No action needed for these
                Case KfxActionValue.KFX_REL_BATCHCLASS_RENAME, _
                    KfxActionValue.KFX_REL_DOCCLASS_RENAME, _
                    KfxActionValue.KFX_REL_TABLE_DELETE, _
                    KfxActionValue.KFX_REL_TABLE_INSERT, _
                    KfxActionValue.KFX_REL_TABLE_RENAME, _
                    KfxActionValue.KFX_REL_UNDEFINED_ACTION, _
                    KfxActionValue.KFX_REL_RELEASESETUP_DELETE, _
                    KfxActionValue.KFX_REL_IMPORT

                Case Else
                    ' Allows for KC adding new events.  Don't fail in release build.
                    Debug.Fail("Unexpected event type in action event")
            End Select

		Catch ex As Exception
			UnexpectedExceptionForm.HandleUnexpectedException(ex)
			eReturnValue = KfxReturnValue.KFX_REL_ERROR
		End Try
		Return eReturnValue
	End Function
#End Region
#Region "Private functions"
	''' <summary>
	''' OnUpgrade Code
	''' </summary>
	''' <remarks></remarks>
	Private Function OnUpgrade() As KfxReturnValue
		Return KfxReturnValue.KFX_REL_UNSUPPORTED
    End Function
    ''' <summary>
    ''' Called when a KC field has been inserted.  
    ''' Shows the UI to the user.
    ''' </summary>
    ''' <param name="strFieldName"></param>
    ''' <param name="eKfxLinkSourceType"></param>
    ''' <remarks></remarks>
    Private Sub OnFieldInsert(ByVal strFieldName As String, ByVal eKfxLinkSourceType As KfxLinkSourceType)
        m_bDirty = True
        m_bShowUI = True
    End Sub
	''' <summary>
    ''' Called when a KC field has been deleted.  
	''' If one of our saved links is deleted, show the UI to the user.
	''' </summary>
	''' <param name="strFieldName"></param>
	''' <param name="eKfxLinkSourceType"></param>
	''' <remarks></remarks>
	Private Sub OnFieldDelete(ByVal strFieldName As String, ByVal eKfxLinkSourceType As KfxLinkSourceType)
        If (m_oSetupDataNet.ILinksNet.Count > 0) Then
            Dim strMacroName As String = MacroBuilder.BuildMacroName(strFieldName, eKfxLinkSourceType)
            Dim oLink As ILink
            For nLinkIndex As Integer = 1 To m_oSetupDataNet.ILinksNet.Count
                oLink = m_oSetupDataNet.ILinksNet.Item(nLinkIndex)
                If (oLink.Source.IndexOf(strMacroName) >= 0) Then
                    ' Removes the deleted field from the source
                    oLink.Source = oLink.Source.Replace(strMacroName, "")
                    m_strDeletedDestinationID = oLink.Destination
                    m_bDirty = True
                    m_bShowUI = True
                    Exit For
                End If
            Next
        End If
	End Sub
	''' <summary>
	''' Called when a KC field has been renamed, this will update our save links to the new one.
	''' </summary>
	''' <param name="strOldValue">Before renaming value</param>
	''' <param name="strNewValue">After renaming value</param>
	''' <param name="eKfxLinkSourceType">KC field type</param>
	''' <remarks></remarks>
	Private Sub OnFieldRename(ByVal strOldValue As String, ByVal strNewValue As String, _
			 ByVal eKfxLinkSourceType As KfxLinkSourceType)
		Dim oLink As ILink
		If (strOldValue.Equals(strNewValue)) Then Return
		For nLinkIndex As Integer = 1 To m_oSetupDataNet.ILinksNet.Count
			oLink = m_oSetupDataNet.ILinksNet.Item(nLinkIndex)
			If (oLink.Source.Equals(strOldValue) And oLink.SourceType.Equals(eKfxLinkSourceType)) Then
				oLink.Source = strNewValue
			End If
		Next
	End Sub
	
	''' <summary>
    ''' Validate entire setup data before publishing a batch
	''' </summary>
	''' <returns></returns>
	''' <remarks></remarks>
	Private Function OnPublishCheck() As KfxReturnValue
        Dim bRequiredFaxNumberFieldFound As Boolean = False


        Dim strConnectorName As String = CustomPropsUtil.ReadCustomProperty(m_oSetupDataNet.ICustomPropertiesNet, Constants.c_strKeyConnector)
        Dim oLink As ILink
        If strConnectorName.Equals(RightFaxConnector.NAME) Then
            ' MB-CODE 2011-02-15 BEGIN comment-out required FROM field
            'Dim bRequiredFaxFromNumberFieldFound As Boolean = True

            ' Finds the two required index fields.
            For nLinkIndex As Integer = 1 To m_oSetupDataNet.ILinksNet.Count
                oLink = m_oSetupDataNet.ILinksNet.Item(nLinkIndex)
                If (oLink.Destination.Equals(FaxDestination.c_IdFaxNumberField)) Then
                    bRequiredFaxNumberFieldFound = True
                End If
                'If (oLink.Destination.Equals(FaxDestination.c_IdFaxFromField)) Then
                'bRequiredFaxFromNumberFieldFound = True
                'End If
            Next
            'If (Not bRequiredFaxFromNumberFieldFound Or Not bRequiredFaxNumberFieldFound) Then
            If (Not bRequiredFaxNumberFieldFound) Then
                Return KfxReturnValue.KFX_REL_ERROR
            End If
            ' MB-CODE 2011-02-15 END comment-out required FROM field

        Else

            Dim bRequiredFaxFromNumberFieldFound As Boolean = True
            For nLinkIndex As Integer = 1 To m_oSetupDataNet.ILinksNet.Count
                oLink = m_oSetupDataNet.ILinksNet.Item(nLinkIndex)
				If (oLink.Destination.Equals(FaxDestination.c_IdFaxNumberField)) Then
					bRequiredFaxNumberFieldFound = True
				End If
                If (oLink.Destination.Equals(FaxDestination.c_IdFaxFromField)) Then
                    bRequiredFaxFromNumberFieldFound = True
                End If

            Next
            If (Not bRequiredFaxFromNumberFieldFound Or Not bRequiredFaxNumberFieldFound) Then
                Return KfxReturnValue.KFX_REL_ERROR
            End If
        End If



        ' Checks the username and server host/ip
        Dim strPath As String
        strPath = CustomPropsUtil.ReadCustomProperty(m_oSetupDataNet.ICustomPropertiesNet, Constants.c_strKeyPath)
        If (String.IsNullOrEmpty(strPath)) Then
            Return KfxReturnValue.KFX_REL_ERROR
        End If
        ' First checks in the custom property
        Dim strUsername As String = CustomPropsUtil.ReadCustomProperty(m_oSetupDataNet.ICustomPropertiesNet, Constants.c_strKeyUsername)
        ' If the username is empty then checks in the Username of the SetupData
        If (String.IsNullOrEmpty(strUsername)) Then
            strUsername = m_oSetupDataNet.UserName
        End If
        If (String.IsNullOrEmpty(strUsername)) Then
            Return KfxReturnValue.KFX_REL_ERROR
        End If
        Return KfxReturnValue.KFX_REL_SUCCESS
    End Function

#End Region


End Class
