'****************************************************************************
'  (c) Copyright 2009 Kofax, Inc.. All rights reserved.
'  Unauthorized use, duplication or distribution is strictly prohibited.
'****************************************************************************
'
' File: BaseFaxConnector.vb
'
' Purpose: Defines the abstract base class for fax connectors.
'
' ****************************************************************************
Imports System.IO
Imports System.Runtime.InteropServices
Imports System.Diagnostics
Imports Microsoft.Win32
Imports Kofax.FaxRel.Connector.My.Resources
Imports tna_fax_COMLib
Imports Kofax.TAPCommon.TAPLib

''' <summary>
''' Defines the abstract base class for fax connectors with common uses of properties, 
''' functions and procedures.
''' </summary>
''' <remarks></remarks>
Public MustInherit Class BaseFaxConnector
    Implements IConnector
#Region "Member declaration"
    ' Defines the registry key name
    Protected Const cm_strRegKeyPath As String = "HKEY_LOCAL_MACHINE\SOFTWARE\Kofax Image Products\Release Scripts\KCEC Fax"
    ' Defines the registry value name of the TNAFax's dll path.
    Protected Const cm_strDllPathValue As String = "DllPath"
    ' Defines the registry value name of the TNAFax's XSLT create config path.
    Protected Const cm_strXsltCreateConfigValue As String = "XsltCreateConfig"
    ' Defines the maximum number of retries.
    Protected Const cm_nMaxRetriesCount As Integer = 5
    ' Defines the maximum time of retries in minutes.
    Protected Const cm_nMaxRetriesInterval As Integer = 2
    ' Defines the sleep interval between retries, in milisecond.
    Protected Const cm_nMaxRetriesSleepInterval As Integer = 200
    ' Declares the fax interface
    Protected m_oFaxInterface As FaxInterface
    ' Declares the variable which hold the current dll working folder.
    Protected m_strDllPath As String
    ' Declares the variable which hold the TNA_Fax home folder(parrent folder of the Bin folder).
    Protected m_strTNAHomeDir As String
    ' Declares the variable which hold the XSLT Create config file path.
    Protected m_strXsltCreateConfig As String

    ' Flag indicates wherther fax environment is initialized.
    Protected m_bFaxInitialized As Boolean
    ' Flag indicates whether fax connector is started.
    Protected m_bFaxStarted As Boolean
    ' Declares the utility for retrieving configuration from resource.
    Protected m_oKfxUtil As FaxConfigUtil
    ' Declares the variable which hold the fax server name or IP address field.
    Private m_strFaxServerName As String
    ' Declares the variable which hold the user Name field.
    Private m_strUsername As String
    ' Declares the variable which hold the user Password field.
    Private m_strPassword As String
    ' Declares the variable which hold the fax Number field.
    Private m_strFaxNumber As String
    ' Declares the variable which hold the fax From field.
    Private m_strFromField As String
    ' Declares the variable which hold the fax To field.
    Private m_strToField As String
    ' Declares the variable which hold the fax SubjectField field.
    Private m_strSubjectField As String
    ' Declares the variable which hold the fax specific Path field.
    Private m_strPath As String
    ' Declares the variable which hold the from fax number field.
    Private m_strFromFaxNumberField As String

#End Region
    Protected Sub New()
        m_bFaxInitialized = False
        m_bFaxStarted = False
        'm_oFaxInterface = New FaxInterface()
        m_oKfxUtil = New FaxConfigUtil()
        Try
            ' Reads the DllPath and XsltCreateConfig variables value under the REG_KEY_PATH
            m_strDllPath = GetRegistryValue(cm_strRegKeyPath, cm_strDllPathValue, String.Empty)
            m_strXsltCreateConfig = GetRegistryValue(cm_strRegKeyPath, cm_strXsltCreateConfigValue, String.Empty)
            If Not m_strDllPath.EndsWith("\") Then
                m_strDllPath = m_strDllPath + "\"
            End If
            Dim di As DirectoryInfo = New DirectoryInfo(m_strDllPath)
            m_strTNAHomeDir = di.Parent.FullName + "\"
        Catch ex As System.Security.SecurityException
            m_strTNAHomeDir = String.Empty
        End Try
    End Sub
#Region "Commons properties"
    ''' <summary>
    ''' Gets or sets the username value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property Username() As String Implements IConnector.Username
        Get
            Return m_strUsername
        End Get
        Set(ByVal value As String)
            m_strUsername = value
        End Set
    End Property
    ''' <summary>
    ''' Gets or sets the password value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property Password() As String Implements IConnector.Password
        Get
            Return m_strPassword
        End Get
        Set(ByVal value As String)
            m_strPassword = value
        End Set
    End Property
    ''' <summary>
    ''' Gets or sets the specific path value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property Path() As String Implements IConnector.Path
        Get
            Return m_strPath
        End Get
        Set(ByVal value As String)
            Dim strOldPath As String = m_strPath
            m_strPath = value
            ' Makes sure we always shutdown the fax environment
            If Not (String.IsNullOrEmpty(m_strPath)) Then
                If IsFaxStarted AndAlso (String.Compare(strOldPath, m_strPath) <> 0) Then
                    ShutDownFax()
                End If
            End If
        End Set
    End Property
    ''' <summary>
    ''' Gets or sets the 'FaxNumber' field value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property FaxNumberField() As String Implements IConnector.FaxNumberField
        Get
            Return m_strFaxNumber
        End Get
        Set(ByVal value As String)
            m_strFaxNumber = value
            If Not (String.IsNullOrEmpty(m_strFaxNumber)) Then
                ' Removes any leading and trailing blank, hyphen, brace and digit sign characters.
                m_strFaxNumber = m_strFaxNumber.Trim()
                ' Removes any extra following chars
                m_strFaxNumber = m_strFaxNumber.Trim(" `~!@#{}[]$%^&*-()_=|\<>/""';".ToCharArray())
            End If
        End Set
    End Property

    ''' <summary>
    ''' Gets or sets the 'FromFaxNumber' field value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property FromFaxNumberField() As String Implements IConnector.FromFaxNumberField
        Get
            Return m_strFromFaxNumberField
        End Get
        Set(ByVal value As String)
            m_strFromFaxNumberField = value
            If Not (String.IsNullOrEmpty(m_strFromFaxNumberField)) Then
                m_strFromFaxNumberField = m_strFromFaxNumberField.Trim()
                m_strFromFaxNumberField = m_strFromFaxNumberField.Trim(" `~!@#{}[]$%^&*-()_=|\<>/""';".ToCharArray())
            End If
        End Set
    End Property

    ''' <summary>
    ''' Gets or sets the 'From' field value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property FromField() As String Implements IConnector.FromField
        Get
            Return m_strFromField
        End Get
        Set(ByVal value As String)
            m_strFromField = value
        End Set
    End Property
    ''' <summary>
    ''' Gets or sets the 'To' field value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property ToField() As String Implements IConnector.ToField
        Get
            Return m_strToField
        End Get
        Set(ByVal value As String)
            m_strToField = value
        End Set
    End Property
    ''' <summary>
    ''' Gets or sets the 'SubjectField' field value.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property SubjectField() As String Implements IConnector.SubjectField
        Get
            Return m_strSubjectField
        End Get
        Set(ByVal value As String)
            m_strSubjectField = value
        End Set
    End Property
    ''' <summary>
    ''' Indicates whether the 'From' field is supported by the fax connector.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public ReadOnly Property IsFromFieldSupported() As Boolean Implements IConnector.IsFromFieldSupported
        Get
            Return True
        End Get
    End Property

    ''' <summary>
    ''' Determines whether the fax connector is initialized.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public ReadOnly Property IsFaxInitialized() As Boolean Implements IConnector.IsFaxInitialized
        Get
            Return m_bFaxInitialized
        End Get
    End Property
    ''' <summary>
    ''' Determines whether the fax connector is started.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public ReadOnly Property IsFaxStarted() As Boolean Implements IConnector.IsFaxStarted
        Get
            Return m_bFaxStarted
        End Get
    End Property
#End Region
#Region "Must override properties/methods/functions for derived classes"
    MustOverride ReadOnly Property InternalName() As String _
        Implements IConnector.InternalName
    ''' <summary>
    ''' Indicates whether the specific path is required.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Result as Boolean.</returns>
    ''' <remarks></remarks>
    Public MustOverride ReadOnly Property IsPathRequired() As Boolean _
        Implements IConnector.IsPathRequired
    ''' <summary>
    ''' Indicates whether the 'To' field is supported and printed on the target fax header.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Result as Boolean.</returns>
    ''' <remarks></remarks>
    Public MustOverride ReadOnly Property IsToFieldSupported() As Boolean _
        Implements IConnector.IsToFieldSupported
    ''' <summary>
    ''' Indicates whether the 'SubjectField' field is supported and printed on the target fax header.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Result as Boolean.</returns>
    ''' <remarks></remarks>
    Public MustOverride ReadOnly Property IsSubjectFieldSupported() As Boolean _
        Implements IConnector.IsSubjectFieldSupported
    ''' <summary>
    ''' Builds the raw configuration specific to the fax connector.
    ''' </summary>
    ''' <param name="strPath">Specific path value.</param>
    ''' <param name="strUsername">The username to include in the configuration.</param>
    ''' <param name="strPassword">The password to include in the configuration.</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public MustOverride Function BuildConfiguration(ByVal strPath As String, _
                                                     ByVal strUsername As String, ByVal strPassword As String) As String _
        Implements IConnector.BuildConfiguration
    ''' <summary>
    ''' Overrides the Object.ToString method to return the fax connector name.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public MustOverride Overrides Function ToString() As String Implements IConnector.ToString
#End Region

#Region "Public Overridable subs/functions"
    ''' <summary>
    ''' Initializes the fax environment.
    ''' </summary>
    ''' <exception cref="InitializeFaxException">If the COM wrapper throws any exception.</exception>
    ''' <remarks>This method loads the C++ Fax interface from the tna_fax.dll into memory. 
    ''' This method needs to be called before any fax operation calls.</remarks>
    Public Overridable Sub InitializeFax() Implements IConnector.InitializeFax
        If (m_bFaxInitialized = True) Then
            Exit Sub
        End If
        Try
            WriteToEventLog("InitializeFax", InternalName, EventLogEntryType.Information)
            m_oFaxInterface = New FaxInterface()
            ' Calls the fax interface wrapper to load the tna_fax.dll into memory
            m_oFaxInterface.InitializeFax()
            m_bFaxInitialized = True
        Catch cex As COMException
            Logger.Log(cex, False, Logger.TraceLevel.Error)
            WriteToEventLog(cex.ToString(), InternalName, EventLogEntryType.Error)
            Throw New InitializeFaxException(Math.Abs(cex.ErrorCode), cex.Message)
        End Try
    End Sub
    ''' <summary>
    ''' Un-Initializes the fax environment. Unload the tna_fax.dll from memory.
    ''' </summary>
    ''' <remarks></remarks>
    Public Overridable Sub DeInitializeFax() Implements IConnector.DeInitializeFax
        ' Checks if the fax has already been un-initialized
        If Not m_bFaxInitialized Then
            Exit Sub
        End If
        WriteToEventLog("DeInitializeFax", InternalName, EventLogEntryType.Information)
        ' First checks if the fax is shut-down
        Try
            If (m_bFaxStarted) Then
                ShutDownFax()
            End If
        Catch cex As COMException
            Logger.Log(cex, False, Logger.TraceLevel.Error)
        End Try
        
        Try
            '*** Now de-initializes the fax
            m_oFaxInterface.DeInitializeFax()
        Catch cex As COMException
            Logger.Log(cex, False, Logger.TraceLevel.Error)
            WriteToEventLog(cex.ToString(), InternalName, EventLogEntryType.Error)
            Throw New DeInitializeFaxException(Math.Abs(cex.ErrorCode), cex.Message)
        Finally
            '*** Release reference to the COM object
            System.Runtime.InteropServices.Marshal.FinalReleaseComObject(m_oFaxInterface)
            m_oFaxInterface = Nothing
            m_bFaxInitialized = False
        End Try
    End Sub
    ''' <summary>
    ''' Tests the login information against the fax server using the current user credentials.
    ''' </summary>
    ''' <returns>Boolean value indicating whether the user can be logged in.</returns>
    ''' <remarks></remarks>
    Public Overridable Function Connect() As Boolean Implements IConnector.Connect
        If Not (m_bFaxInitialized) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been initialized.")
        End If
        If Not (m_bFaxStarted) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been started.")
        End If
        Return TestServer(Path, Username, Password)
    End Function
    ''' <summary>
    ''' Tests the login information against the fax server using the specified server name and user credentials.
    ''' </summary>
    ''' <param name="strFaxServerName">The fax server name.</param>
    ''' <param name="strUsername">The username to test.</param>
    ''' <param name="strPassword">The password to test.</param>
    ''' <returns>Boolean value indicating whether the user can be logged in.</returns>
    ''' <remarks></remarks>
    Public Overridable Function Connect(ByVal strFaxServerName As String, _
                                          ByVal strUsername As String, ByVal strPassword As String) As Boolean _
        Implements IConnector.Connect
        If Not (m_bFaxInitialized) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been initialized.")
        End If
        If Not (m_bFaxStarted) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been started.")
        End If
        Return TestServer(strFaxServerName, strUsername, strPassword)
    End Function
    ''' <summary>
    ''' Starts the fax interface with the associated information.
    ''' </summary>
    ''' <remarks>This method can be overridden for a specific fax connector.</remarks>
    Public Overridable Sub StartFax() Implements IConnector.StartFax
        If (m_bFaxStarted = True) Then
            Exit Sub
        End If
        If Not (m_bFaxInitialized) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been initialized.")
        End If
        Try
            Dim strRawConfig As String = BuildConfiguration(Path, Username, Password)
            Logger.Log(String.Format("Start the fax interface with configuration string: {0}", strRawConfig), 0, False, Logger.TraceLevel.Info)
            WriteToEventLog(String.Format("StartFax: TnaHome: {0}; Xslt Config: {1}; Raw Config: {2}", m_strTNAHomeDir, m_strXsltCreateConfig, strRawConfig), InternalName, EventLogEntryType.Information)
            m_oFaxInterface.Start(m_strTNAHomeDir, strRawConfig, m_strXsltCreateConfig)
            m_bFaxStarted = True
        Catch cex As COMException
            Logger.Log(cex, False, Logger.TraceLevel.Error)
            WriteToEventLog(cex.ToString(), InternalName, EventLogEntryType.Error)
            Throw New FacsimileException(Math.Abs(cex.ErrorCode), cex.Message)
        End Try
    End Sub
    ''' <summary>
    ''' Sends the list of TIFF images to the fax connector.
    ''' </summary>
    ''' <param name="oListImages"></param>
    ''' <remarks>This method can be overridden for a specific fax connector.</remarks>
    Public Overridable Sub SendFax(ByVal oListImages As IList) Implements IConnector.SendFax
        WriteToEventLog(String.Format("SendFax: FaxNumberField={0}; FromFaxNumberField={1}", FaxNumberField, FromFaxNumberField), InternalName, EventLogEntryType.Information)
        If Not (m_bFaxInitialized) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been initialized.")
        End If
        If Not (m_bFaxStarted) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been started.")
        End If
        If String.IsNullOrEmpty(FaxNumberField) Then
            ' Destination fax number is not specified.
            Throw New FacsimileException(Resources.ERROR_FAX_NUMBER_NOT_SPECIFIED)
        End If
        If Not IsValidFaxNumber(FaxNumberField) Then
            ' Destination fax number is not valid.
            Throw New FacsimileException(Resources.ERROR_INVALID_FAX_NUMBER)
        End If
        If String.IsNullOrEmpty(FromFaxNumberField) Then
            ' The fax from number is not specified.
            Throw New FacsimileException(Resources.ERROR_FAX_FROM_NUMBER_NOT_SPECIFIED)
        End If
        If Not IsValidFaxNumber(FromFaxNumberField) Then
            ' The fax from number contains characters that are not supported.
            Throw New FacsimileException(Resources.ERROR_INVALID_FAX_FROM_NUMBER)
        End If
        ' Calculates the estimated time for retries
        Dim dtmEndTime As DateTime = DateTime.Now.AddMinutes(cm_nMaxRetriesInterval)
        Dim nRetries As Integer = 0
        Dim bLoop As Boolean = True
        While bLoop
            nRetries = nRetries + 1
            Try
                WriteToEventLog(String.Format("Sending fax with loop #{0}", nRetries), InternalName, EventLogEntryType.Information)
                InternalSendFax(oListImages)
                ' The fax has been successfully sent. Set the flag to exit the loop
                bLoop = False
            Catch ex As Exception
                If TypeOf ex Is RetriableException Then
                    If (dtmEndTime.CompareTo(System.DateTime.Now) < 0) Or (nRetries > cm_nMaxRetriesCount) Then
                        Throw
                    Else
                        ' Blocks the current thread for the specified miliseconds
                        Threading.Thread.Sleep(Math.Max(200, cm_nMaxRetriesSleepInterval))
                        Continue While
                    End If
                Else
                    ' Some other exception or Unretry able exception occurred so exit the loop immediately 
                    ' and re-throws the exception.
                    Throw
                End If
            End Try
        End While
    End Sub
    ''' <summary>
    ''' Shuts down the fax after uses.
    ''' </summary>
    ''' <remarks>This method can be overridden based on the specific fax connector.</remarks>
    Public Overridable Sub ShutDownFax() Implements IConnector.ShutdownFax
        If Not (m_bFaxStarted) Then
            Exit Sub
        End If
        Try
            WriteToEventLog("ShutDownFax", InternalName, EventLogEntryType.Information)
            ' Shuts down the fax interface
            m_oFaxInterface.Shutdown(20000)
            m_bFaxStarted = False
        Catch cex As COMException
            Logger.Log(cex, False, Logger.TraceLevel.Error)
            WriteToEventLog(cex.ToString(), InternalName, EventLogEntryType.Error)
            Throw New FacsimileException(Math.Abs(cex.ErrorCode), cex.Message)
        End Try
    End Sub
    ''' <summary>
    ''' Verifies that the appropriate client software is installed.
    ''' </summary>
    ''' <remarks>Base class does not verify any client installed. Derived class should override this method if requires any client installed.</remarks>
    Public Overridable Sub VerifyClientInstalled() Implements IConnector.VerifyClientInstalled
    End Sub
#End Region

#Region "Protected overridable subs/functions"
    ''' <summary>
    ''' This procedure allows any derived class to do something pre-test the server using 
    ''' the specified username and password against the fax server.
    ''' </summary>
    ''' <param name="strPath">The server name or IP address or the path share.</param>
    ''' <param name="strUserName">The user username.</param>
    ''' <param name="strPassword">The user password.</param>
    ''' <returns>Boolean.True if login to the target fax server succeeds.</returns>
    ''' <remarks></remarks>
    Protected Overridable Sub PreTestServer(ByVal strPath As String, ByVal strUserName As String, ByVal strPassword As String)

    End Sub
    ''' <summary>
    ''' Tests the username and password against the fax server.
    ''' </summary>
    ''' <param name="strPath">The server name or IP address or the path share.</param>
    ''' <param name="strUserName">The user username.</param>
    ''' <param name="strPassword">The user password.</param>
    ''' <returns>Boolean.True if login to the target fax server succeeds.</returns>
    ''' <remarks></remarks>
    Protected Overridable Function TestServer(ByVal strPath As String, ByVal strUserName As String, ByVal strPassword As String) As Boolean
        If (String.IsNullOrEmpty(strPath)) Then
            ' Invalid fax server name or IP address.
            Throw New InvalidFaxServerNameException(Resources.ERROR_FAX_SERVER_NOT_SPECIFIED)
        ElseIf (String.IsNullOrEmpty(strUserName)) Then
            ' The fax's username is not specified.
            Throw New FacsimileException(Resources.ERROR_FAX_USER_NOT_SPECIFIED)
        ElseIf m_bFaxInitialized = False Then
            ' Facsimile environment is not initialized.
            Throw New FacsimileException(Resources.ERROR_FAX_NOT_INITIALIZED)
        ElseIf m_bFaxStarted = False Then
            ' Fax connector is not started.
            Throw New FacsimileException(Resources.ERROR_FAX_NOT_STARTED)
        End If
        Try
            PreTestServer(strPath, strUserName, strPassword)
            Dim oServerProps As IServerProperties
            oServerProps = m_oFaxInterface.GetFaxServerProperties(strUserName, strPassword)
            ' Release reference to the COM object
            System.Runtime.InteropServices.Marshal.FinalReleaseComObject(oServerProps)
            Return True
        Catch ex As COMException
            Logger.Log(ex, False, Logger.TraceLevel.Error)
            WriteToEventLog(ex.ToString(), InternalName, EventLogEntryType.Error)
            Dim msg As String = ex.Message
            Throw New FacsimileException(Math.Abs(ex.ErrorCode), msg)
        End Try
    End Function
    ''' <summary>
    ''' Sends the images to the fax server.
    ''' </summary>
    ''' <param name="oListImages"></param>
    ''' <exception cref="FacsimileException">If any error occurred during the sendFax operation.</exception>
    ''' <remarks></remarks>
    Protected Overridable Sub InternalSendFax(ByVal oListImages As IList)
        Dim strExpectedCSI As String = String.Empty
        Dim strSourceTSI As String = CStr(IIf(String.IsNullOrEmpty(FromFaxNumberField), "", FromFaxNumberField))
        Dim e As IEnumerator = oListImages.GetEnumerator()
        Try
            While (e.MoveNext())
                Dim strTiff As String = CStr(e.Current)
                m_oFaxInterface.SendFaxFromFile(Username, Password, strTiff, FaxNumberField, strExpectedCSI, ToField, strSourceTSI, FromField, SubjectField)
            End While
        Catch cex As COMException
            Logger.Log(cex, False, Logger.TraceLevel.Error)
            WriteToEventLog(cex.ToString(), InternalName, EventLogEntryType.Error)
            Dim errorCode As Integer = Math.Abs(cex.ErrorCode)
            ' Customizes the exception
            ' The following error codes and messages are thrown by the COM wrapper
            '1. Fax Interface has not been initialized.
            '2. Fax interface has already been initialized.
            '3. Failed to load library tna_fax.dll.
            '4. Failed to get Fax Interface.
            '5. Fax Interface has not been started.
            '6. Fax Interface has already been started.
            '7. The XSLT configuration file could not be found.
            ' Error codes other than 1..7 are thrown by the TNA_Fax C++ fax interface.
            Dim nErrorNumber As Integer = Math.Abs(cex.ErrorCode)
            Dim strErrorMessage As String = cex.Message
            Select Case nErrorNumber
                Case 2, 4, 5, 6
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 1, 3, 4, 7
                    Throw New UnRetriableException(nErrorNumber, strErrorMessage)
                Case 558891526 ' KCS error code - Error 558891526: 612 server temporarily busy (Cat=C_ErrObjNotExist, Obj=Connection, Mod=M_Tcsi)
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 586163469 ' RightFax error code - Error 586163469: Login to fax server(rightfax) failed. Error: The remote procedure call failed. (Cat=C_ErrAuthentication, Obj=TncRFax, Mod=M_RFax)
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 586163982 ' Biscom error code - Error 586163982: Login to fax server(\\sqa-biscom\FaxcomQ) failed. Error: Invalid user type for this login method. (Cat=C_ErrAuthentication, Obj=TncBiscom, Mod=M_Biscom)
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 537922305 ' Error 537922305: Error by fileToObj 1028 (Cat=C_Error, Obj=XMLParser, Mod=M_IntellNetwork).
                    Logger.Log(strErrorMessage, nErrorNumber, False, Logger.TraceLevel.Error)
                    ' Customized message: Cannot release the document to the fax because the document contains image types other than TIFF.
                    Throw New UnRetriableException(nErrorNumber, Resources.ERROR_INVALID_TIFF_FORMAT)
                Case Else
                    Throw New UnRetriableException(nErrorNumber, strErrorMessage)
            End Select
        Catch ex As Exception
            WriteToEventLog(ex.ToString(), InternalName, EventLogEntryType.Error)
            Throw New UnRetriableException(Integer.MaxValue, ex.Message)
        End Try
    End Sub
    ''' <summary>
    ''' Validates the fax number
    ''' </summary>
    ''' <param name="strNumber"></param>
    ''' <returns></returns>
    ''' <remarks>Apply a simple syntactical check following: <br/><ul><li>Allowed characters are: 0123456789#*-+() </li><li>The number must not be empty.</li></ul></remarks>
    Protected Overridable Function IsValidFaxNumber(ByVal strNumber As String) As Boolean
        Dim strAllowedFaxNum As String = " +-0123456789#"
        ' Checks
        Dim strNumChars As Char() = strNumber.Trim().ToCharArray()
        For Each c As Char In strNumChars
            If (strAllowedFaxNum.IndexOf(c) < 0) Then
                Return False
            End If
        Next
        ' The number has the plus sign but not at the beginning
        If (strNumber.IndexOf("+") >= 0) And Not strNumber.StartsWith("+") Then
            Return False
        End If
        ' The number has the hyphen sign but at the beginning
        If (strNumber.IndexOf("-") >= 0) And strNumber.StartsWith("-") Then
            Return False
        End If
        ' The number has the digit sign but at the beginning
        If (strNumber.IndexOf("#") >= 0) And strNumber.StartsWith("#") Then
            Return False
        End If
        ' TODO: other character such as [*()]?
        Return True
    End Function
    ''' <summary>
    ''' Checks whether a file exists.
    ''' </summary>
    ''' <param name="strFileName">The file name to check.</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Protected Overridable Function IsFileExist(ByVal strFileName As String) As Boolean
        If String.IsNullOrEmpty(strFileName) Then
            Return False
        End If
        Dim oFileInfo As FileInfo = New FileInfo(strFileName)
        Return oFileInfo.Exists
    End Function

    ''' <summary>
    ''' Retrieves the value associated with the specified name, in the specified registry key. 
    ''' If the name is not found in the specified key, returns a default value that you provide, 
    ''' or null if the specified key does not exist.
    ''' </summary>
    ''' <param name="strKeyPath">The name of the key path.</param>
    ''' <param name="strValueName">The name of the name/value pair.</param>
    ''' <param name="strDefaultValue">The value to return if name does not exist.</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Protected Overridable Function GetRegistryValue(ByVal strKeyName As String, ByVal strValueName As String, ByVal strDefaultValue As String) As String
        Try
            Return String.Format("{0}", Registry.GetValue(strKeyName, strValueName, strDefaultValue))
        Catch ex As Security.SecurityException
            Return Nothing
        Catch ex As Exception
            Return Nothing
        End Try
    End Function

    ''' <summary>
    ''' Writes an entry to Event Log. 
    ''' </summary>
    ''' <param name="strEntry">The Value to write to event log.</param>
    ''' <param name="strAppName">Name of Client Application. Needed because before writing to event log, 
    ''' you must have a named EventLog source.</param>
    ''' <param name="eEventType">Entry Type, from EventLogEntryType structure e.g., EventLogEntryType.Warning, 
    ''' EventLogEntryType.Error</param>
    ''' <param name="strLogName">Name of Log (System, Application, Security is read-only) If you specify a 
    ''' non-existent log, the log will be created.</param>
    ''' <returns>True if successful, false if not.</returns>
    ''' <remarks>EventSources are tightly tied to their log. So don't use the same source name for different 
    ''' logs, and vice versa.</remarks>
    Protected Function WriteToEventLog(ByVal strEntry As String, _
        Optional ByVal strAppName As String = "Application", _
        Optional ByVal eEventType As EventLogEntryType = EventLogEntryType.Information, _
        Optional ByVal strLogName As String = "Application") As Boolean
        Dim objEventLog As New EventLog()
        Try
            ' Registers the App as an Event Source
            If Not EventLog.SourceExists(strAppName) Then
                EventLog.CreateEventSource(strAppName, strLogName)
            End If
            objEventLog.Source = strAppName
            ' WriteEntry is overloaded; this is one
            'of 10 ways to call it
            objEventLog.WriteEntry(strEntry, eEventType)
            Return True
        Catch Ex As Exception
            Return False
        Finally
            objEventLog.Close()
        End Try
    End Function
#End Region
End Class
