'****************************************************************************
'  (c) Copyright 2009 Kofax, Inc.. All rights reserved.
'  Unauthorized use, duplication or distribution is strictly prohibited.
'****************************************************************************
'
' File: BiscomFAXCOMConnector.vb
'
' Purpose: Defines the specific implementation for Biscom-FAXCOM fax connector.
'
' ****************************************************************************
Imports Kofax.FaxRel.Connector.My.Resources
Imports System.IO
Public Class BiscomFAXCOMConnector
    Inherits BaseFaxConnector

    Public Sub New()
        MyBase.New()
        ' Detects the Biscom client config from the Registry
        Dim strFaxServerName As String = GetRegistryValue("HKEY_CURRENT_USER\Software\Biscom\FaxComApi\1.0", "FaxServerName", String.Empty)
        If Not String.IsNullOrEmpty(strFaxServerName) Then
            Path = strFaxServerName
        End If
    End Sub
    ''' <summary>
    ''' Returns the internal implementation name of the connector.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides ReadOnly Property InternalName() As String
        Get
            Return "BISCOM_FAXCOM_FAX"
        End Get
    End Property
    ''' <summary>
    ''' Overrides the Object.ToString method to return the fax connector name.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ToString() As String
		Return Resources.CONNECTOR_BISCOM
    End Function
    ''' <summary>
    ''' Indicates whether the 'Path' field is required.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Boolean.True since the path value is required for the Biscom fax connector.</returns>
    ''' <remarks></remarks>
    Public Overrides ReadOnly Property IsPathRequired() As Boolean
        Get
            Return True
        End Get
    End Property
    ''' <summary>
    ''' Indicates whether the 'To' field is supported on the target fax header.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Boolean.True since the 'To' field is supported by the Biscom fax connector.</returns>
    ''' <remarks></remarks>
    Public Overrides ReadOnly Property IsToFieldSupported() As Boolean
        Get
            Return True
        End Get
    End Property
    ''' <summary>
    ''' Indicates whether the 'SubjectField' field is supported on the target fax header.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Boolean.False since the 'SubjectField' field is not supported by the Biscom fax connector.</returns>
    ''' <remarks></remarks>
    Public Overrides ReadOnly Property IsSubjectFieldSupported() As Boolean
        Get
            Return False
        End Get
    End Property
    ''' <summary>
    ''' Builds the raw configuration specific to the BISCOM FAXCOM fax connector.
    ''' </summary>
    ''' <param name="strPath"></param>
    ''' <param name="strUsername"></param>
    ''' <param name="strPassword"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function BuildConfiguration(ByVal strPath As String, ByVal strUsername As String, ByVal strPassword As String) As String
        Dim strConfigName As String = m_oKfxUtil.CONFIG_BISCOM
        Dim strRawConfig As String = m_oKfxUtil.GetConfig(strConfigName)
        If Not String.IsNullOrEmpty(strRawConfig) Then
            ' Replaces the pre-defined string "<Path>0.0.0.0</Path>" with the network share path of the BISCOM server share.
            strRawConfig = strRawConfig.Replace("<Path>0.0.0.0</Path>", String.Format("<Path>{0}</Path>", strPath))
        End If
        Return strRawConfig
    End Function
    ''' <summary>
    ''' Verifies that the appropriate client software is installed.
    ''' </summary>
    ''' <exception cref="RequireClientMissingException">If any required client is not installed.</exception>
    ''' <remarks></remarks>
    Public Overrides Sub VerifyClientInstalled()
        ' Checks for the following registry key and existence of file based its value.
        ' HKEY_LOCAL_MACHINE\SOFTWARE\Biscom\FAXCOM Manager for value FaxcomManagerPath (Client version: 09.01.0612, 09.04.0100)
        ' and also HKEY_LOCAL_MACHINE\SOFTWARE\Biscom\FAXCOM Suite for Windows Client\Lan for value BisFMPath (Client version: 09.01.0612, 09.04.0100)
        ' HKEY_LOCAL_MACHINE\SOFTWARE\Biscom\FAXCOM Suite for Windows Client value InstallPath  09.04.0100
        Dim strBisFMPath As String = GetRegistryValue("HKEY_LOCAL_MACHINE\SOFTWARE\Biscom\FAXCOM Suite for Windows Client\Lan", "BisFMPath", String.Empty)
        Dim strFaxcomManagerPath As String = GetRegistryValue("HKEY_LOCAL_MACHINE\SOFTWARE\Biscom\FAXCOM Manager", "FaxcomManagerPath", String.Empty)
        If (String.IsNullOrEmpty(strBisFMPath)) Or (String.IsNullOrEmpty(strFaxcomManagerPath)) Then
            Throw New RequireClientMissingException(Resources.ERROR_FAXCOM_CLIENT_MISSING)
        End If
        Try
            Dim oBisFMFileInfo As FileInfo = New FileInfo(strBisFMPath)
            Dim oFaxcomManagerFileInfo As FileInfo = New FileInfo(strFaxcomManagerPath)
            If (Not oBisFMFileInfo.Exists) Or (Not oFaxcomManagerFileInfo.Exists) Then
                Throw New FileNotFoundException()
            End If
        Catch ex As Exception
            Throw New RequireClientMissingException(Resources.ERROR_FAXCOM_CLIENT_MISSING)
        End Try
    End Sub
    
End Class
