'****************************************************************************
'  (c) Copyright 2009 Kofax, Inc.. All rights reserved.
'  Unauthorized use, duplication or distribution is strictly prohibited.
'****************************************************************************
'
' File: Release.vb
'
' Purpose: Defines the release facade for fax connectors.
'
' ****************************************************************************
Imports System.IO
Imports System.Diagnostics
Imports System.Collections.Generic
Imports Kofax.ReleaseLib
Imports Kofax.TAPCommon.TAPLib
Imports Kofax.TAPCommon.TAPEncryption
Imports System.Text.RegularExpressions
Imports Kofax.FaxRel.Connector.My.Resources

''' <summary>
''' Defines the release facade for fax connectors.
''' </summary>
''' <remarks></remarks>


Public Class Release
    ' Tiff file filter pattern
    Private Const cm_strTiffPattern As String = "^\.tiff$|\.tif$"
    ' Declares an object which represents the ReleaseData object.
    Private m_oReleaseData As IReleaseDataNet
    ' Declares the crypto utility object.
    Private m_oCrypto As AscentDataCrypto
    ' Declares the temporary release working folder.
    Private m_strReleaseWorkingFolder As String = String.Empty
    ' Declares the fax connector object.
    Private m_oIConnector As IConnector
    ' Declares a flag which represents the started state of the fax connector.
    Private m_bFaxStarted As Boolean

    ' MB-CODE 2011-01-26 BEGIN
    Private Structure RightFaxCustomFields
        Dim BillingInfo1 As String
        Dim BillingInfo2 As String
        Dim CoversheetMode As String
        Dim Owner As String
        Dim SecurePassword As String
        Dim DeliveryMode As String
    End Structure
    ' MB-CODE 2011-01-26 END
    ''' <summary>
    ''' Constructor
    ''' </summary>
    ''' <param name="releaseData">The IReleaseDataNet wrapper of the ReleaseData.</param>
    ''' <remarks></remarks>
    <CLSCompliant(False)> _
    Public Sub New(ByVal releaseData As IReleaseDataNet)
        Try
            ' Initializes the crypto object.
            m_oCrypto = New AscentDataCrypto(releaseData)
            ' Saves the release data.
            m_oReleaseData = releaseData
            ' Creates new instance of the connector
            InitializeConnector()
            m_bFaxStarted = False
        Catch ex As Exception
            WriteToEventLog(ex.ToString(), "FaxRelease", EventLogEntryType.Error)
            Throw
        End Try
    End Sub
    ''' <summary>
    ''' Releases documents.
    ''' </summary>
    ''' <param name="releaseData">Release data object.</param>
    ''' <returns>True if successful, false otherwise.</returns>
    ''' <remarks></remarks>
    <CLSCompliant(False)> _
    Public Function ReleaseDoc(ByVal releaseData As IReleaseDataNet) As KfxReturnValue
        Dim returnValue As Integer = 0

        Try
            ' SPR00048946: Verifies that the appropriate client software is installed.
            m_oIConnector.VerifyClientInstalled()
            If Not m_oIConnector.IsFaxInitialized Then
                Throw New FacsimileException(Resources.ERROR_FAX_NOT_INITIALIZED)
            End If
            m_oReleaseData = releaseData
            ' IImageFilesNet includes edocs.  The only way the count can be zero is if there is only
            ' one image and skip first page is true and there are no edocs.
            If m_oReleaseData.IImageFilesNet.Count = 0 Then
                ' No page to export.
                WriteToEventLog(Resources.ERROR_NOPAGE_TO_EXPORT, "FaxRelease", EventLogEntryType.Error)
                Throw New FacsimileException(Resources.ERROR_NOPAGE_TO_EXPORT)
            End If
            ' Gets release working folder saved in the KC db + root working folder + unique subfolder
            m_strReleaseWorkingFolder = CreateReleaseWorkingFolder()
            ' Copies documents to the release working folder
            If CopyDocument() Then
                ' Initializes the fax setting with default value.
                Dim strToFaxes As String = String.Empty
                Dim strFromFax As String = String.Empty
                Dim strSubject As String = String.Empty
                Dim strTo As String = String.Empty
                Dim strFrom As String = String.Empty
                Dim oKCVarKeyValues As Hashtable = New Hashtable()
                Dim oKCIndexKeyValues As Hashtable = New Hashtable()
                Dim oKCBatchKeyValues As Hashtable = New Hashtable()


                ' MB-CODE 2011-01-26 begin
                Dim stRightFaxCustomValues As New RightFaxCustomFields
                ' MB-CODE 2011-01-26 END

                ' BEGIN SPR00096368, using FOR loop and Using statement instead of WHILE loop and Dispose()
                Dim nValueCount As Integer
                For nValueCount = 1 To m_oReleaseData.IValuesNet.Count
                    Using valueNet As IValueNet = m_oReleaseData.IValuesNet.Item(nValueCount)

                        If String.Equals(FaxDestination.c_IdFaxNumberField, valueNet.Destination) Then
                            strToFaxes = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdFaxFromField, valueNet.Destination) Then
                            strFromFax = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdSubjectField, valueNet.Destination) Then
                            strSubject = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdToField, valueNet.Destination) Then
                            strTo = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdSenderField, valueNet.Destination) Then
                            strFrom = valueNet.Value
                            ' MB-CODE 2011-01-26 BEGIN
                        ElseIf String.Equals(FaxDestination.c_IdOwnerField, valueNet.Destination) Then
                            stRightFaxCustomValues.Owner = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdBilling1Field, valueNet.Destination) Then
                            stRightFaxCustomValues.BillingInfo1 = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdBilling2Field, valueNet.Destination) Then
                            stRightFaxCustomValues.BillingInfo2 = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdSecurePasswordField, valueNet.Destination) Then
                            stRightFaxCustomValues.SecurePassword = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdDeliveryModeField, valueNet.Destination) Then
                            stRightFaxCustomValues.DeliveryMode = valueNet.Value
                        ElseIf String.Equals(FaxDestination.c_IdCoversheetModeField, valueNet.Destination) Then
                            stRightFaxCustomValues.CoversheetMode = valueNet.Value
                            ' MB-CODE 2011-01-26 END
                        End If

                        ' Saves each variable, index field or batch field with its value
                        Select Case valueNet.SourceType
                            Case KfxLinkSourceType.KFX_REL_INDEXFIELD
                                oKCIndexKeyValues.Add(valueNet.Destination, valueNet.Value)
                            Case KfxLinkSourceType.KFX_REL_BATCHFIELD
                                oKCBatchKeyValues.Add(valueNet.Destination, valueNet.Value)
                            Case KfxLinkSourceType.KFX_REL_VARIABLE
                                oKCVarKeyValues.Add(valueNet.Destination, valueNet.Value)
                        End Select
                    End Using
                Next
                ' END SPR00096368
                ' Deals with multi-macro
                ' Replace all macros for strToFaxes with priority index, batch fields and KC variables 
                strToFaxes = ReplaceAllMacroWithValue(strToFaxes, oKCIndexKeyValues, oKCBatchKeyValues, oKCVarKeyValues)
                strFromFax = ReplaceAllMacroWithValue(strFromFax, oKCIndexKeyValues, oKCBatchKeyValues, oKCVarKeyValues)
                strSubject = ReplaceAllMacroWithValue(strSubject, oKCIndexKeyValues, oKCBatchKeyValues, oKCVarKeyValues)
                strFrom = ReplaceAllMacroWithValue(strFrom, oKCIndexKeyValues, oKCBatchKeyValues, oKCVarKeyValues)
                strTo = ReplaceAllMacroWithValue(strTo, oKCIndexKeyValues, oKCBatchKeyValues, oKCVarKeyValues)
                oKCIndexKeyValues.Clear()
                oKCIndexKeyValues = Nothing
                oKCBatchKeyValues.Clear()
                oKCBatchKeyValues = Nothing
                oKCVarKeyValues.Clear()
                oKCVarKeyValues = Nothing
                ' Assigns value to the connector instance
                m_oIConnector.FromFaxNumberField = strFromFax
                m_oIConnector.SubjectField = strSubject
                m_oIConnector.FromField = strFrom
                m_oIConnector.ToField = strTo

                ' MB-CODE 2011-01-26 BEGIN
                ' if this is a Rightfax pointer, then add these values to the new properties
                If TypeOf m_oIConnector Is RightFaxConnector Then
                    Dim oRightFaxConn As RightFaxConnector
                    oRightFaxConn = DirectCast(m_oIConnector, RightFaxConnector)
                    oRightFaxConn.Owner = stRightFaxCustomValues.Owner
                    oRightFaxConn.Billing1 = stRightFaxCustomValues.BillingInfo1
                    oRightFaxConn.Billing2 = stRightFaxCustomValues.BillingInfo2
                    oRightFaxConn.SecurePassword = stRightFaxCustomValues.SecurePassword
                    oRightFaxConn.DeliveryMode = stRightFaxCustomValues.DeliveryMode
                    oRightFaxConn.CoversheetMode = stRightFaxCustomValues.CoversheetMode
                End If
                ' MB-CODE 2011-01-26 END

                Dim oDirInfo As DirectoryInfo = New DirectoryInfo(m_strReleaseWorkingFolder)
                If oDirInfo.Exists Then
                    Dim arrFileInfoArr As FileInfo() = oDirInfo.GetFiles()
                    ' Declares the list of tiff file names.
                    Dim oListFiles As ArrayList = New ArrayList()
                    ' Collects all the tiff or tif files.
                    Dim oFileInfo As FileInfo
                    For Each oFileInfo In arrFileInfoArr
                        Dim strNameLower As String = oFileInfo.Extension.ToLower()
                        If (Regex.IsMatch(strNameLower, cm_strTiffPattern, Text.RegularExpressions.RegexOptions.IgnoreCase)) Then
                            oListFiles.Add(oFileInfo.FullName)
                        End If
                    Next oFileInfo
                    ' Starts the fax environment
                    m_oIConnector.StartFax()
                    m_bFaxStarted = True
                    Dim arrSendingFaxNum() As String = strToFaxes.Split(",;".ToCharArray())
                    If Not (arrSendingFaxNum Is Nothing) And (arrSendingFaxNum.Length > 0) Then
                        Dim en As IEnumerator = arrSendingFaxNum.GetEnumerator()
                        While (en.MoveNext())
                            Dim strFaxToSend As String = CStr(en.Current)
                            If Not (String.Equals(String.Empty, strFaxToSend)) Then
                                m_oIConnector.FaxNumberField = strFaxToSend
                                ' Sends the fax with the list of tiff files.
                                m_oIConnector.SendFax(oListFiles)
                            End If
                        End While
                    End If
                    returnValue = KfxReturnValue.KFX_REL_SUCCESS
                End If
            End If
        Catch ex As Exception
            ' Unexpected Error: {0}
            WriteToEventLog(ex.ToString(), "FaxRelease", EventLogEntryType.Error)
            Dim errorMessage As String = String.Format(Resources.ERROR_UNEXPECTED_ERROR, ex.Message)
            'If TypeOf ex Is FacsimileException Then
            'Logger.Log(errorMessage, CType(ex, FacsimileException).ErrorCode, False, Logger.TraceLevel.Error)
            ' Else
            'Logger.Log(errorMessage, 0, False, Logger.TraceLevel.Error)
            'End If
            'Logger.Log(ex, False, Logger.TraceLevel.Error)
            m_oReleaseData.SendMessage(errorMessage, 0, KfxInfoReturnValue.KFX_REL_DOC_ERROR)
            returnValue = KfxReturnValue.KFX_REL_FATALBATCHERROR
        Finally
            RemoveRootWorkingFolder(m_strReleaseWorkingFolder)
            If (m_bFaxStarted) Then
                ' Shuts down the fax environment
                m_oIConnector.ShutdownFax()
                m_bFaxStarted = False
            End If
        End Try
        Return CType(returnValue, KfxReturnValue)
    End Function
    ''' <summary>
    ''' Initializes the fax environment.
    ''' </summary>
    ''' <remarks>This method should be called in OpenScript function.</remarks>
    Public Sub InitializeFax()
        ' Initializes the fax interface.
        Try
            'MB-CODE 2011-02-16 BEGIN Add version of Assembly to startup logging
            Dim sVersion As String = String.Empty
            sVersion = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString()
            WriteToEventLog("Initialize Fax, assembly version: " & sVersion, "InitializeFax", EventLogEntryType.Information)
            'MB-CODE 2011-02-16 END Add version of Assembly to startup logging

            m_oIConnector.InitializeFax()
        Catch ex As FacsimileException
            'Logger.Log(ex.ToString(), ex.ErrorCode, False, Logger.TraceLevel.Error)
            WriteToEventLog(ex.ToString(), "FaxRelease", EventLogEntryType.Error)
            Throw
        Catch ex As System.Exception
            'Logger.Log(ex.Message, -1, False, Logger.TraceLevel.Error)
            Throw
        End Try
    End Sub
    ''' <summary>
    ''' De-initializes the fax environment.
    ''' </summary>
    ''' <remarks>This method should be called in CloseScript function.</remarks>
    Public Sub DeInitializeFax()
        Try
            If Not (m_oIConnector Is Nothing) Then
                m_oIConnector.DeInitializeFax()
            End If
        Catch ex As FacsimileException
            'Logger.Log(ex.ToString(), ex.ErrorCode, False, Logger.TraceLevel.Error)
            WriteToEventLog(ex.ToString(), "FaxRelease", EventLogEntryType.Error)
            Throw
        Catch ex As Exception
            'Logger.Log(ex.Message, -1, False, Logger.TraceLevel.Error)
            Throw
        End Try
    End Sub
    ''' <summary>
    ''' Removes the root working folder.
    ''' </summary>
    ''' <remarks>This method should be called in CloseScript function.</remarks>
    Public Sub RemoveRootWorkingFolder(ByVal folderToRemove As String)
        Try
            ' Removes the temp folder
            If folderToRemove <> String.Empty Then
                GeneralUtil.TAPRemoveDirectory(folderToRemove)
            End If
        Catch ex As Exception
            ' We don't want to fail the doc over this
            'Logger.Log(ex.Message, False, Logger.TraceLevel.Warning)
        End Try
    End Sub

    ''' <summary>
    ''' Copies the document's images that are being faxed out.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function CopyDocument() As Boolean
        With m_oReleaseData
            .IImageFilesNet.Copy(m_strReleaseWorkingFolder, AC_IMAGE_TYPE.MTIFF_G4)
        End With
        Return True
    End Function
    ''' <summary>
    ''' Builds the path of the root working folder.
    ''' </summary>
    ''' <returns>The path of the root working folder</returns>
    ''' <remarks>This starts with the working folder saved in Setup and appends the working folder to that.</remarks>
    Private Function GetRootWorkingFolder() As String
        Return m_oReleaseData.ImageFilePath
    End Function
    ''' <summary>
    ''' Creates a release working folder from the root working folder by appending
    '''  a unique subfolder
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function CreateReleaseWorkingFolder() As String
        Dim rootWorkingFolder As String = GetRootWorkingFolder()
        ' Creates a unique subfolder
        Return GeneralUtil.TAPCreateUniqueSubDir(rootWorkingFolder)
    End Function

    Private Function IsValidInfo() As Boolean
        ' Gets the export connector name
        Dim strConnectorName As String = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyConnector)
        If (String.Compare(m_oIConnector.InternalName, strConnectorName) <> 0) Then
            Return False
        End If
        ' Reads the username value from custom property storage.
        Dim strSavedUsername As String = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyUsername)
        If Not (String.IsNullOrEmpty(strSavedUsername)) Then
            strSavedUsername = m_oCrypto.Decrypt(strSavedUsername)
        End If
        If (String.Compare(m_oIConnector.Username, strSavedUsername) <> 0) Then
            Return False
        End If
        ' Reads the user password value from custom property storage.
        Dim strSavedPassword As String = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyPassword)
        If Not (String.IsNullOrEmpty(strSavedPassword)) Then
            strSavedPassword = m_oCrypto.Decrypt(strSavedPassword)
        End If
        If (String.Compare(m_oIConnector.Password, strSavedPassword) <> 0) Then
            Return False
        End If
        ' Reads the specific PATH value from custom property storage.
        Dim strPath As String = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyPath)
        If (String.Compare(m_oIConnector.Path, strPath) <> 0) Then
            Return False
        End If
        Return True
    End Function

    Private Sub InitializeConnector()
        ' Gets the export connector name
        Dim strConnectorName As String = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyConnector)
        m_oIConnector = ConnectorFactory.GetConnectorByName(strConnectorName)
        If m_oIConnector Is Nothing Then
            ' Could not retrieve the export connector from the release configuration.
            Dim strMessage As String = Resources.ERROR_RETRIEVE_CONNECTOR
            'Logger.Log(strMessage, 0, False, Logger.TraceLevel.Error)
            Throw New FacsimileException(strMessage)
        End If
        ' Initializes the username to the export connector.
        m_oIConnector.Username = String.Empty
        ' Decrypts the username setting in the release script's setting.
        Dim strSavedUsername As String = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyUsername)
        ' Tries to get username from SetupData if we can not retrieve username from the custom property
        If (String.IsNullOrEmpty(strSavedUsername)) Then
            strSavedUsername = m_oReleaseData.UserName
        End If
        If Not (String.IsNullOrEmpty(strSavedUsername)) Then
            m_oIConnector.Username = m_oCrypto.Decrypt(strSavedUsername)
        End If
        ' Initializes the user password.
        m_oIConnector.Password = String.Empty
        ' Decrypts the password setting in the release script's setting.
        Dim strSavedPassword As String = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyPassword)
        ' Tries to get password from SetupData if we can not retrieve password from the custom property
        If (String.IsNullOrEmpty(strSavedPassword)) Then
            strSavedPassword = m_oReleaseData.Password
        End If
        If Not (String.IsNullOrEmpty(strSavedPassword)) Then
            m_oIConnector.Password = m_oCrypto.Decrypt(strSavedPassword)
        End If
        ' Reads the specific PATH value from custom property storage.
        m_oIConnector.Path = CustomPropsUtil.ReadCustomProperty(m_oReleaseData.ICustomPropertiesNet, Constants.c_strKeyPath)
    End Sub

    Private Function WriteToEventLog(ByVal Entry As String, _
           Optional ByVal AppName As String = "Application", _
           Optional ByVal EventType As EventLogEntryType = EventLogEntryType.Information, _
           Optional ByVal LogName As String = "Application") As Boolean
        Dim objEventLog As New EventLog()
        Try
            ' Registers the App as an Event Source
            If Not EventLog.SourceExists(AppName) Then
                EventLog.CreateEventSource(AppName, LogName)
            End If
            objEventLog.Source = AppName
            ' WriteEntry is overloaded; this is one
            'of 10 ways to call it
            objEventLog.WriteEntry(Entry, EventType)
            Return True
        Catch Ex As Exception
            Return False
        Finally
            objEventLog.Close()
        End Try
    End Function
    ''' <summary>
    ''' Replaces all occurrence of key by its value.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function ReplaceMacroWithValue(ByVal strMultiMacroSubject As String, ByVal oKeyValues As Hashtable) _
        As String
        Dim strResult As String = strMultiMacroSubject
        If Not (IsContainMacro(strResult)) Then
            Return strResult
        End If
        Dim key As String
        Dim value As String
        Dim en As IDictionaryEnumerator = oKeyValues.GetEnumerator()
        While en.MoveNext
            ' Checks whether we should replace or not to increase the performance.
            If (IsContainMacro(strResult)) Then
                key = CStr(en.Key)
                value = CStr(en.Value)
                strResult = strResult.Replace(key, value)
            Else
                ' No more macroes to replace, returns it.
                Return strResult
            End If
        End While
        Return strResult
    End Function
    ''' <summary>
    ''' Indicates whether the subject contains macroes.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function IsContainMacro(ByVal strSubject As String) As Boolean
        If (String.IsNullOrEmpty(strSubject)) Then
            Return False
        End If
        Return Regex.IsMatch(strSubject, ".*{[@|\$]?\S.*}.*", RegexOptions.IgnoreCase)
    End Function
    ''' <summary>
    ''' Replaces all occurrence of key by its value.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function ReplaceAllMacroWithValue(ByVal strMultiMacroSubject As String, ByVal oIndexKeyValues As Hashtable, _
                                               ByVal oBatchKeyValues As Hashtable, _
                                               ByVal oVarKeyValues As Hashtable) As String
        Dim strResult As String = ReplaceMacroWithValue(strMultiMacroSubject, oIndexKeyValues)
        strResult = ReplaceMacroWithValue(strResult, oBatchKeyValues)
        strResult = ReplaceMacroWithValue(strResult, oVarKeyValues)
        Return strResult
    End Function
End Class
