'****************************************************************************
'  (c) Copyright 2009 Kofax, Inc.. All rights reserved.
'  Unauthorized use, duplication or distribution is strictly prohibited.
'****************************************************************************
'
' File: RightFaxConnector.vb
'
' Purpose: Defines the specific implementation for the Captaris RightFax fax connector.
'
' ****************************************************************************
Imports System.IO
Imports Kofax.FaxRel.Connector.My.Resources
Imports Kofax.TAPCommon.TAPLib
Imports System.Runtime.InteropServices

''' <summary>
''' Defines the specific implementation for the Captaris RightFax fax connector.
''' </summary>
''' <remarks></remarks>
Public Class RightFaxConnector

    Inherits BaseFaxConnector
    Public Shared NAME As String = "CAPTARIS_RIGHTFAX_FAX"
    ' MB-CODE 2011-01-26 BEGIN
    ' new fields for new Rightfax functionality
    Private m_Owner As String = String.Empty
    Private m_CoversheetMode As String = String.Empty
    Private m_Billing1 As String = String.Empty
    Private m_Billing2 As String = String.Empty
    Private m_DeliveryMode As String = String.Empty
    Private m_SecurePassword As String = String.Empty
    'MB-CODE 2011-01-26 END

    Public Sub New()
        MyBase.New()
        ' Detects the current configured RightFax server name from the Registry
        Dim strRFServerName As String = GetRegistryValue("HKEY_LOCAL_MACHINE\SOFTWARE\RightFax\Install", "RFServerName", String.Empty)
        If Not String.IsNullOrEmpty(strRFServerName) Then
            Path = strRFServerName
        End If
    End Sub
    ''' <summary>
    ''' Returns the internal implementation name of the connector.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides ReadOnly Property InternalName() As String
        Get
            Return NAME
        End Get
    End Property
    ''' <summary>
    ''' Overrides the method ToString of the Object to return the fax connector name.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ToString() As String
        Return Resources.CONNECTOR_RIGHTFAX
    End Function
    ''' <summary>
    ''' Indicates whether the 'Path' field is required.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Boolean.True since the 'Path' field is required by the RightFax fax connector.</returns>
    ''' <remarks></remarks>
    Public Overrides ReadOnly Property IsPathRequired() As Boolean
        Get
            Return True
        End Get
    End Property
    ''' <summary>
    ''' Indicates whether the 'To' field is supported on the target fax header.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Boolean.True since the 'To' field is supported by the RightFax fax connector.</returns>
    ''' <remarks></remarks>
    Overrides ReadOnly Property IsToFieldSupported() As Boolean
        Get
            Return True
        End Get
    End Property
    ''' <summary>
    ''' Indicates whether the 'SubjectField' field is supported on the target fax header.
    ''' </summary>
    ''' <value></value>
    ''' <returns>Boolean.False since the 'SubjectField' field is not supported by the RightFax fax connector.</returns>
    ''' <remarks></remarks>
    Public Overrides ReadOnly Property IsSubjectFieldSupported() As Boolean
        Get
            Return False
        End Get
    End Property
    ''' <summary>
    ''' Builds the raw configuration specific to the RightFax connector.
    ''' </summary>
    ''' <param name="strPath">Specific path value. Required.</param>
    ''' <param name="strUsername">The username to include in the configuration. Required.</param>
    ''' <param name="strPassword">The password to include in the configuration. Required.</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function BuildConfiguration(ByVal strPath As String, ByVal strUsername As String, ByVal strPassword As String) As String
        Dim strConfigName As String = m_oKfxUtil.CONFIG_CAPTARIS
        Dim strRawConfig As String = m_oKfxUtil.GetConfig(strConfigName)
        If Not String.IsNullOrEmpty(strRawConfig) Then
            ' Replaces the pre-defined string "<Path>0.0.0.0</Path>" with the server name or IP address of the RightFax server.
            strRawConfig = strRawConfig.Replace("<Path>0.0.0.0</Path>", String.Format("<Path>{0}</Path>", strPath))
        End If
        Return strRawConfig
    End Function
    ''' <summary>
    ''' Verifies that the appropriate client software is installed.
    ''' </summary>
    ''' <remarks></remarks>
    Public Overrides Sub VerifyClientInstalled()
        ' Checks registry key HKEY_LOCAL_MACHINE\SOFTWARE\RightFax\Install for value InstallDir
        ' Checks existence of the client directory and required files.
        ' Also HKEY_LOCAL_MACHINE\SOFTWARE\RightFax Client for value CodesFile and FaxCtrlCmd
        Dim strInstallDir As String = GetRegistryValue("HKEY_LOCAL_MACHINE\SOFTWARE\RightFax\Install", "InstallDir", String.Empty)

        'MB-CODE 2011-02-24 BEGIN -- comment out these checks, the client version 9.4 doesn't even have "CodesFile" key
        'Dim strCodesFile As String = GetRegistryValue("HKEY_LOCAL_MACHINE\SOFTWARE\RightFax Client", "CodesFile", String.Empty)
        'Dim strFaxCtrlCmd As String = GetRegistryValue("HKEY_LOCAL_MACHINE\SOFTWARE\RightFax Client", "FaxCtrlCmd", String.Empty)

        'If (String.IsNullOrEmpty(strInstallDir)) Or (String.IsNullOrEmpty(strCodesFile)) Or (String.IsNullOrEmpty(strFaxCtrlCmd)) Then
        If (String.IsNullOrEmpty(strInstallDir)) Then
            Throw New RequireClientMissingException(Resources.ERROR_FAXCOM_CLIENT_MISSING)
        End If
        Try
            Dim oInstallDirDirectoryInfo As DirectoryInfo = New DirectoryInfo(strInstallDir)
            'Dim oCodesFileFileInfo As FileInfo = New FileInfo(strCodesFile)
            'Dim oFaxCtrlCmdFileInfo As FileInfo = New FileInfo(strFaxCtrlCmd)
            'If (Not oInstallDirDirectoryInfo.Exists) Or (Not oCodesFileFileInfo.Exists) Or (Not oFaxCtrlCmdFileInfo.Exists) Then

            If (Not oInstallDirDirectoryInfo.Exists) Then
                Throw New FileNotFoundException("Rightfax install directory: " & strInstallDir & " doesn't exist!")
            End If
            'MB-CODE 2011-02-24 END -- comment out these checks
        Catch ex As Exception
            Throw New RequireClientMissingException(Resources.ERROR_RIGHTFAX_CLIENT_MISSING)
        End Try
    End Sub

    ''' <summary>
    ''' Sends the images to the fax server.
    ''' </summary>
    ''' <param name="oListImages"></param>
    ''' <exception cref="FacsimileException">If any error occurred during the sendFax operation.</exception>
    ''' <remarks>This is an override of the base InternalSendFax to send to new Rightfax Interface.</remarks>
    ' MB-CODE 2011-01-26 BEGIN
    Protected Overrides Sub InternalSendFax(ByVal oListImages As IList)
        Dim strExpectedCSI As String = String.Empty
        Dim strSourceTSI As String = CStr(IIf(String.IsNullOrEmpty(FromFaxNumberField), "", FromFaxNumberField))
        Dim e As IEnumerator = oListImages.GetEnumerator()
        Try
            While (e.MoveNext())
                Dim strTiff As String = CStr(e.Current)
                'call the new COM dll interface to the new C++ dll interface, etc.
                m_oFaxInterface.SendFaxFromFileExt(Username, Password, strTiff, FaxNumberField, strExpectedCSI, ToField, strSourceTSI, FromField, SubjectField, CoversheetMode, Owner, Billing1, Billing2, DeliveryMode, SecurePassword)
            End While
            ' MB-CODE 2011-01-26 END

        Catch cex As COMException
            Logger.Log(cex, False, Logger.TraceLevel.Error)
            WriteToEventLog(cex.ToString(), InternalName, EventLogEntryType.Error)
            Dim errorCode As Integer = Math.Abs(cex.ErrorCode)
            ' Customizes the exception
            ' The following error codes and messages are thrown by the COM wrapper
            '1. Fax Interface has not been initialized.
            '2. Fax interface has already been initialized.
            '3. Failed to load library tna_fax.dll.
            '4. Failed to get Fax Interface.
            '5. Fax Interface has not been started.
            '6. Fax Interface has already been started.
            '7. The XSLT configuration file could not be found.
            ' Error codes other than 1..7 are thrown by the TNA_Fax C++ fax interface.
            Dim nErrorNumber As Integer = Math.Abs(cex.ErrorCode)
            Dim strErrorMessage As String = cex.Message
            Select Case nErrorNumber
                Case 2, 4, 5, 6
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 1, 3, 4, 7
                    Throw New UnRetriableException(nErrorNumber, strErrorMessage)
                Case 558891526 ' KCS error code - Error 558891526: 612 server temporarily busy (Cat=C_ErrObjNotExist, Obj=Connection, Mod=M_Tcsi)
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 586163469 ' RightFax error code - Error 586163469: Login to fax server(rightfax) failed. Error: The remote procedure call failed. (Cat=C_ErrAuthentication, Obj=TncRFax, Mod=M_RFax)
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 586163982 ' Biscom error code - Error 586163982: Login to fax server(\\sqa-biscom\FaxcomQ) failed. Error: Invalid user type for this login method. (Cat=C_ErrAuthentication, Obj=TncBiscom, Mod=M_Biscom)
                    Throw New RetriableException(nErrorNumber, strErrorMessage)
                Case 537922305 ' Error 537922305: Error by fileToObj 1028 (Cat=C_Error, Obj=XMLParser, Mod=M_IntellNetwork).
                    Logger.Log(strErrorMessage, nErrorNumber, False, Logger.TraceLevel.Error)
                    ' Customized message: Cannot release the document to the fax because the document contains image types other than TIFF.
                    Throw New UnRetriableException(nErrorNumber, Resources.ERROR_INVALID_TIFF_FORMAT)
                Case Else
                    Throw New UnRetriableException(nErrorNumber, strErrorMessage)
            End Select
        Catch ex As Exception
            WriteToEventLog(ex.ToString(), InternalName, EventLogEntryType.Error)
            Throw New UnRetriableException(Integer.MaxValue, ex.Message)
        End Try
    End Sub

    ' MB-CODE 2011-01-28 BEGIN - Add new properties for RightFax

    ''' <summary>
    ''' This will be converted to an enum value (int)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property CoversheetMode() As String
        Get
            CoversheetMode = m_CoversheetMode
        End Get
        Set(ByVal value As String)
            m_CoversheetMode = MapCoversheetMode(value)
        End Set
    End Property

    ''' <summary>
    ''' 
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Property Owner() As String
        Get
            Owner = m_Owner
        End Get
        Set(ByVal value As String)
            m_Owner = value
        End Set
    End Property

    ''' <summary>
    ''' Billing1 value
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Property Billing1() As String
        Get
            Billing1 = m_Billing1
        End Get
        Set(ByVal value As String)
            m_Billing1 = value
        End Set
    End Property

    ''' <summary>
    ''' Billing2 value
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Property Billing2() As String
        Get
            Billing2 = m_Billing2

        End Get
        Set(ByVal value As String)
            m_Billing2 = value
        End Set
    End Property

    ''' <summary>
    ''' Password if secure doc
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Property SecurePassword() As String
        Get
            SecurePassword = m_SecurePassword
        End Get
        Set(ByVal value As String)
            m_SecurePassword = value
        End Set
    End Property

    ''' <summary>
    ''' This will be converted to an enum value (int)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Property DeliveryMode() As String
        Get
            DeliveryMode = m_DeliveryMode
        End Get
        Set(ByVal value As String)
            m_DeliveryMode = MapDeliveryMode(value)
        End Set
    End Property

    ' MB-CODE 2011-01-31 BEGIN
    ''' <summary>
    ''' Maps Kofax string value to Rightfax values.
    ''' </summary>
    ''' <param name="strInputValue">Kofax string value</param>
    ''' <returns>Mapped Rightfax value.</returns>
    ''' <remarks></remarks>
    Private Function MapCoversheetMode(ByVal strInputValue As String) As String
        Try
            Select Case UCase(strInputValue)
                Case "NO"
                    Return "0" 'NO COVERSHEET
                Case "SYSTEMDEFAULT"
                    Return "1" 'SYSTEM DEFAULT BEHAVIOR
                Case "YES"
                    Return "2" 'USE COVERSHEET
                Case Else
                    Throw New NoMappedValue(Resources.ERROR_RIGHTFAX_NO_MAPPED_VALUE + ", Input: " & strInputValue)
            End Select
        Catch ex As Exception
            Throw ex
        End Try

    End Function

    ' MB-CODE 2011-01-31 BEGIN
    ''' <summary>
    ''' Maps Kofax string value to Rightfax values.
    ''' </summary>
    ''' <param name="strInputValue">Kofax string value</param>
    ''' <returns>Mapped Rightfax value.</returns>
    ''' <remarks></remarks>
    Private Function MapDeliveryMode(ByVal strInputValue As String) As String
        Try
            Select Case UCase(strInputValue)
                Case "FAX"
                    Return "0" ' SEND AS FAX
                Case "SECUREDOC"
                    Return "1" 'SEND AS SECURE DOC (NEED PASSWORD)
                Case "CERTIFIEDDOC"
                    Return "3" 'SEND AS CERTIFIED DOC
                Case Else
                    Throw New NoMappedValue(Resources.ERROR_RIGHTFAX_NO_MAPPED_VALUE + ", Input: " & strInputValue)
            End Select
        Catch ex As Exception
            Throw ex
        End Try

    End Function
    'MB-CODE 2011-02-18 BEGIN - OVERRIDE BASE FUNCTIONALITY TO NOT CHECK FOR FAX NUMBERS IF IT'S NOT A FAX
    ''' <summary>
    ''' Sends the list of TIFF images to the fax connector.
    ''' </summary>
    ''' <param name="oListImages"></param>
    ''' <remarks>This method overrides the base method for RightFax functions.</remarks>
    Public Overrides Sub SendFax(ByVal oListImages As IList)
        WriteToEventLog(String.Format("SendFax: FaxNumberField={0}; FromFaxNumberField={1}", FaxNumberField, FromFaxNumberField), InternalName, EventLogEntryType.Information)
        If Not (m_bFaxInitialized) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been initialized.")
        End If
        If Not (m_bFaxStarted) Then
            Throw New FacsimileException("Illegal Fax's state operation: Fax environment has not been started.")
        End If

        If DeliveryMode = "0" Then 'its a fax, verify fax number

            If String.IsNullOrEmpty(FaxNumberField) Then
                ' Destination fax number is not specified.
                Throw New FacsimileException(Resources.ERROR_FAX_NUMBER_NOT_SPECIFIED)
            End If

            If Not IsValidFaxNumber(FaxNumberField) Then
                ' Destination fax number is not valid.
                Throw New FacsimileException(Resources.ERROR_INVALID_FAX_NUMBER)
            End If
        Else
            'Do nothing, it's not a Fax, it's an email address
        End If

        ' Calculates the estimated time for retries
        Dim dtmEndTime As DateTime = DateTime.Now.AddMinutes(cm_nMaxRetriesInterval)
        Dim nRetries As Integer = 0
        Dim bLoop As Boolean = True
        While bLoop
            nRetries = nRetries + 1
            Try
                WriteToEventLog(String.Format("Sending fax with loop #{0}", nRetries), InternalName, EventLogEntryType.Information)
                InternalSendFax(oListImages)
                ' The fax has been successfully sent. Set the flag to exit the loop
                bLoop = False
            Catch ex As Exception
                If TypeOf ex Is RetriableException Then
                    If (dtmEndTime.CompareTo(System.DateTime.Now) < 0) Or (nRetries > cm_nMaxRetriesCount) Then
                        Throw
                    Else
                        ' Blocks the current thread for the specified miliseconds
                        Threading.Thread.Sleep(Math.Max(200, cm_nMaxRetriesSleepInterval))
                        Continue While
                    End If
                Else
                    ' Some other exception or Unretry able exception occurred so exit the loop immediately 
                    ' and re-throws the exception.
                    Throw
                End If
            End Try
        End While
    End Sub
    'MB-CODE 2011-02-18 END - OVERRIDE BASE FUNCTIONALITY TO NOT CHECK FOR FAX NUMBERS IF IT'S NOT A FAX


    ' MB-CODE 2011-01-26 END
End Class
