'****************************************************************************
'  (c) Copyright 2009 Kofax, Inc.. All rights reserved.
'  Unauthorized use, duplication or distribution is strictly prohibited.
'****************************************************************************
'
' File: SetupForm.vb
'
' Purpose: Display the fax server setup UI.
'
' ****************************************************************************
Imports System.Windows.Forms
Imports Kofax.TAPCommon.TAPControls
Imports Kofax.FaxRel.Connector.My.Resources
Imports Kofax.TAPCommon.TAPLib
Imports Kofax.ReleaseLib

''' <summary>
''' Display the fax server setup UI.
''' </summary>
''' <remarks></remarks>
Public Class SetupForm
    ' This enumeration is defined from Microsoft.VisualBasic.Compatibility.VB6.ShiftConstants
    Private Enum ShiftConstants As Integer
        ShiftMask = 1
        CtrlMask = 2
        AltMask = 4
    End Enum

#Region "Members"
    Private m_oSetup As Setup
    Private m_strHelpPath As String
    Private m_DeletedDestinationID As String
    Private m_bFieldsDirty As Boolean
    Private m_bFormShown As Boolean
#End Region

#Region "Public sub/functions"
    ''' <summary>
    ''' Creates new instance of SetupForm class and associate setup class object to it.
    ''' </summary>
    ''' <param name="setup">An instance of the setup object.</param>
    ''' <remarks></remarks>
    Public Sub New(ByVal setup As Setup)
        ' This call is required by the Windows Form Designer.
        InitializeComponent()
        ' Add any initialization after the InitializeComponent() call.
        m_oSetup = setup
        m_strHelpPath = GeneralUtil.GetHelpFilePath(Constants.c_strHelpFileName)
        m_DeletedDestinationID = Nothing
        m_bFieldsDirty = False
    End Sub

    ''' <summary>
    ''' Show the export connector setup dialog. Set focus to Fax Settings tab if oActionEventHandler is not null
    ''' </summary>
    ''' <param name="oActionEventHandler">An instance of ActinoEventHandler class</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overloads Function ShowDialog(Optional ByVal oActionEventHandler As ActionEventHandler = Nothing) As DialogResult
        If Not (oActionEventHandler Is Nothing) Then
            ctlTabSetup.SelectedTab = ctlTbPgFields
            m_DeletedDestinationID = oActionEventHandler.DeletedDestinationID
            ' Enables the Apply button incase new fields inserted or field deleted.
            m_bFieldsDirty = oActionEventHandler.IsDirty
        End If
        Return MyBase.ShowDialog()
    End Function
#End Region

#Region "Propeties"

    ''' <summary>
    ''' Gets the instance of Setup class
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public ReadOnly Property ClsSetup() As Setup
        Get
            Return m_oSetup
        End Get
    End Property

    ''' <summary>
    ''' Gets the current instance of IConnector
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public ReadOnly Property SelectedConnector() As IConnector
        Get
            Return GetCurrentConnectorFromComboBox()
        End Get
    End Property
    ''' <summary>
    ''' Gets or sets the dirty state of the setup data.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Property IsDirty() As Boolean
        Get
            Return m_oSetup.IsDirty
        End Get
        Set(ByVal value As Boolean)
            m_oSetup.IsDirty = value
            ' Sets the apply enabled/disabled
            cmdApply.Enabled = value
        End Set
    End Property

#End Region

#Region "Private subs/functions"
    Class LinkRow
        Private m_strText As String
        Private m_eSourceType As KfxLinkSourceType
        Private m_strID As String
        Public Sub New(ByVal strText As String, ByVal eSourceType As KfxLinkSourceType, ByVal strID As String)
            Text = strText
            SourceType = eSourceType
            ID = strID
        End Sub
        Public Property Text() As String
            Get
                Return m_strText
            End Get
            Set(ByVal value As String)
                m_strText = value
            End Set
        End Property
        Public Property SourceType() As KfxLinkSourceType
            Get
                Return m_eSourceType
            End Get
            Set(ByVal value As KfxLinkSourceType)
                m_eSourceType = value
            End Set
        End Property
        Public Property ID() As String
            Get
                Return m_strID
            End Get
            Set(ByVal value As String)
                m_strID = value
            End Set
        End Property
    End Class
    ''' <summary>
    ''' Return the current instace of IConnector
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function GetCurrentConnectorFromComboBox() As IConnector
        Return CType(cboFaxServer.SelectedItem, IConnector)
    End Function


    ''' <summary>
    ''' Handles when form is being close by clicking Cancel, Alt-F4, X button.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function ConfirmChanges() As Boolean
        If (m_oSetup.IsDirty) Then
            Dim eDlgResult As DialogResult

            ' Asks for saving change before closing the form       
            eDlgResult = MessageBox.Show(Resources.MSG_WARNING_CANCEL_SAVE, _
                    Resources.COMMON_SHORT_TITLE, _
                    MessageBoxButtons.YesNoCancel, _
                    MessageBoxIcon.Question)
            If (eDlgResult = System.Windows.Forms.DialogResult.Yes) Then
                Return SaveSetupData()
            ElseIf (eDlgResult = System.Windows.Forms.DialogResult.No) Then
                m_oSetup.IsDirty = False
                Return True
            Else
                Return False
            End If
        Else
            Return True
        End If
    End Function

    ''' <summary>
    ''' Verifies and save user settings to KC database. 
    ''' No user settings will be saved if the data is invalid.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks>
    ''' </remarks>
    Private Function SaveSetupData() As Boolean
        Dim strMsg As String = String.Empty
        Dim bRetValue As Boolean = True
        Try
            'Sets the form values to Setup object instance
            m_oSetup.SetupDataNet.Name = txtName.Text.Trim()
            With m_oSetup.ClsFaxServerSetup
                .InternalConnectorName = GetCurrentConnectorFromComboBox().InternalName
                .Path = txtPath.Text.Trim()
                .Username = txtUsername.Text.Trim()
                .Password = txtPassword.Text.Trim()
            End With
            'Verifies the setup data
            m_oSetup.Verify()
            ctlLnkIndexFields.Verify()
            'Saves the setup data for TAP Links
            With m_oSetup.SetupDataNet
                .ILinksNet.RemoveAll()
                For Each row As TAPLinksRow In ctlLnkIndexFields.Rows
                    .ILinksNet.Add(row.Source.Text, row.Source.SourceType, row.Destination.ID)
                Next
                AddAssignedVariablesOrIndexes(.ILinksNet)
            End With
            ' Checks the Index Fields for any not assigned, only document index fields and batch fields are checked, 
            ' folder index fields are skipped.
            bRetValue = VerifyIndexFieldsUsage()
        Catch ex As InvalidUsernameException
            ctlTabSetup.SelectTab(Constants.TabPageIndexEnum.FaxServerTab)
            txtUsername.Focus()
            bRetValue = False
            strMsg = ex.Message
        Catch ex As InvalidPathException
            ctlTabSetup.SelectTab(Constants.TabPageIndexEnum.FaxServerTab)
            txtPath.Focus()
            bRetValue = False
            strMsg = ex.Message
        Catch ex As RequiredAttributeMissingException
            ctlTabSetup.SelectTab(Constants.TabPageIndexEnum.FaxSettingsTab)
            Dim oTAPLnkRow As TAPLinksRow = ctlLnkIndexFields.GetRowByID(ex.OffendingItemID)
            If Not (oTAPLnkRow Is Nothing) Then
                oTAPLnkRow.Focus()
            End If
            bRetValue = False
            strMsg = String.Format(Resources.ERROR_EMPTY_FIELD, ex.OffendingItemName)
        Finally
            If Not bRetValue Then
                MessageBox.Show(strMsg, Resources.COMMON_SHORT_TITLE, MessageBoxButtons.OK, MessageBoxIcon.Exclamation)
            End If
            ' Now saves the setup data
            If (bRetValue) Then
                'Saves the setup data
                m_oSetup.SaveSetupData()
            End If
        End Try
        Return bRetValue
    End Function

    Private Function BuildFormatedMessage(ByVal fields() As String, ByVal strPrefix As String) As String
        Dim strFields As String = String.Empty
        If (fields.Length >= 10) Then
            strFields = strPrefix + String.Join(strPrefix, fields, 0, 9) + vbCrLf + vbTab + Resources.MSG_LISTING_ANDMORE
        Else
            strFields = strPrefix + String.Join(strPrefix, fields)
        End If
        Return strFields
    End Function
    ''' <summary>
    ''' Verifies uses of the document and batch index fields. 
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks>
    ''' </remarks>
    Private Function VerifyIndexFieldsUsage() As Boolean
        Try
            Dim arrIndexFields As ArrayList = New ArrayList()
            Dim arrBatchFields As ArrayList = New ArrayList()
            ' Checks the Index Fields for any not assigned
            Using enumerator As New ComEnumerator(Me.m_oSetup.SetupDataNet.IIndexFieldsNet.GetEnumerator())
                While enumerator.MoveNext()
                    Dim oIndexField As IIndexField = CType(enumerator.Current, IIndexField)
                    Using New ComDisposer(oIndexField)
                        ' Checks only document index field, and skip folder index field.
                        If String.IsNullOrEmpty(oIndexField.FolderClassName) Then
                            If Not IsIndexValueUsed(oIndexField.Name, KfxLinkSourceType.KFX_REL_INDEXFIELD) Then
                                arrIndexFields.Add(oIndexField.Name)
                            End If
                        End If
                    End Using
                End While
                enumerator.Dispose()
            End Using
            Using enumerator As New ComEnumerator(Me.m_oSetup.SetupDataNet.IBatchFieldsNet.GetEnumerator())
                While enumerator.MoveNext()
                    Dim oIndexField As IBatchField = CType(enumerator.Current, IBatchField)
                    Using New ComDisposer(oIndexField)
                        ' Checks only document index field, and skip folder index field.
                        If Not IsIndexValueUsed(oIndexField.Name, KfxLinkSourceType.KFX_REL_BATCHFIELD) Then
                            arrBatchFields.Add(oIndexField.Name)
                        End If
                    End Using
                End While
                enumerator.Dispose()
            End Using
            If (arrIndexFields.Count > 0) Or (arrBatchFields.Count > 0) Then
                Dim indFields() As String = CType(arrIndexFields.ToArray(GetType(String)), String())
                Dim batFields() As String = CType(arrBatchFields.ToArray(GetType(String)), String())
                Throw New UnusedFieldsException("Some index fields are not used", indFields, batFields)
			End If
			return True
		Catch ex As UnusedFieldsException
			' Shows warning for document index fields
			Dim oIndexFields() As String = ex.DocumentFields
			Dim retVal As DialogResult = Windows.Forms.DialogResult.None
			Dim strFieldPrefix As String = String.Format("{0}{1}- ", vbCrLf, vbTab)
			Dim strFields As String
			Dim strMsg As String
			If (Not oIndexFields Is Nothing) AndAlso (oIndexFields.Length > 0) Then
				strFields = BuildFormatedMessage(oIndexFields, strFieldPrefix)
				strMsg = String.Format("{0}{1}", Resources.MSG_WARNING_INDEXFIELDS_NOT_USED, strFields)
				retVal = MessageBox.Show(strMsg, Resources.COMMON_SHORT_TITLE, MessageBoxButtons.OKCancel, MessageBoxIcon.Information)
			End If
			' Shows warning for batch fields, if the user click Cancel, then we dismiss this message
			If (retVal = Windows.Forms.DialogResult.None) Or (retVal = Windows.Forms.DialogResult.OK) Then
				Dim oBatchFields() As String = ex.BatchFields
				If (Not oBatchFields Is Nothing) AndAlso (oBatchFields.Length > 0) Then
					strFields = BuildFormatedMessage(oBatchFields, strFieldPrefix)
					strMsg = String.Format("{0}{1}", Resources.MSG_WARNING_BATCHFIELDS_NOT_USED, strFields)
					retVal = MessageBox.Show(strMsg, Resources.COMMON_SHORT_TITLE, MessageBoxButtons.OKCancel, MessageBoxIcon.Information)
				End If
			End If
			If (retVal = Windows.Forms.DialogResult.Cancel) Then
				Return False
			Else
				Return True
			End If
		Catch ex As Exception
			Return False
		End Try
	End Function

    ''' <summary>
    ''' Indicates whether an index field is used in any link. 
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks>
    ''' </remarks>
    Private Function IsIndexValueUsed(ByVal strName As String, ByVal eIndexFieldType As KfxLinkSourceType) _
        As Boolean
        With m_oSetup.SetupDataNet
            Dim strSearchFieldName As String = MacroBuilder.BuildMacroName(strName, eIndexFieldType)
            ' Now iterates the use of the strName in all links setting
            For Each row As TAPLinksRow In ctlLnkIndexFields.Rows
                Dim strDesDisName As String = row.Destination.DisplayName
                Dim strDesText As String = row.Destination.FormattedDisplayName
                Dim strSourceName As String = row.Source.Text
                Dim strSourceText As String = row.Source.DisplayText
                Dim sourceText As String = row.SourceText
                Dim eLinkSourceType As KfxLinkSourceType = row.Source.SourceType
                Select Case eLinkSourceType
                    Case KfxLinkSourceType.KFX_REL_BATCHFIELD
                        If Not String.IsNullOrEmpty(sourceText) AndAlso sourceText.IndexOf(strSearchFieldName) >= 0 Then
                            Return True
                        End If
                    Case KfxLinkSourceType.KFX_REL_INDEXFIELD
                        If Not String.IsNullOrEmpty(sourceText) AndAlso sourceText.IndexOf(strSearchFieldName) >= 0 Then
                            ' We found the strName is used, return true
                            Return True
                        End If
                    Case KfxLinkSourceType.KFX_REL_TEXTCONSTANT
                        If (sourceText.IndexOf(strSearchFieldName) >= 0) Then
                            Return True
                        End If
                        ' Does nothing for undefined and variable cases
                    Case KfxLinkSourceType.KFX_REL_UNDEFINED_LINK
                    Case KfxLinkSourceType.KFX_REL_VARIABLE
                End Select
            Next
        End With
        Return False
    End Function

    ''' <summary>
    ''' Adds all assigned document index fields, batch index fields and KC variables to the Links collection.
    ''' </summary>
    ''' <remarks>This indexes or variables will have value in release runtime.</remarks>
    Private Sub AddAssignedVariablesOrIndexes(ByVal oLinksNet As ILinksNet)
        ' Collects all text fields from taplinks
        ' Iterates entire that collection and add any used variables or document/batch fields
        Dim linkCollection As List(Of LinkRow) = New List(Of LinkRow)
        For Each row As TAPLinksRow In ctlLnkIndexFields.Rows
            linkCollection.Add(New LinkRow(row.Source.Text, row.Source.SourceType, row.Destination.ID))
        Next
		' Iterates entire the Bacth variable names and stores variable to the links object for any variable assigned.
		Dim oLinkDef As LinkRow = Nothing
		Dim enumLink As List(Of LinkRow).Enumerator = Nothing
		Dim vars As TAPCommon.TAPLib.BatchVariableNamesAdapter = CType(m_oSetup.SetupDataNet.BatchVariableNames, BatchVariableNamesAdapter)
		For Each oVarName As String In vars
			Dim strSearchVarName As String = MacroBuilder.BuildMacroName(oVarName, KfxLinkSourceType.KFX_REL_VARIABLE)
			enumLink = linkCollection.GetEnumerator()
			' Iterates entire that collection and add any used variables
			While enumLink.MoveNext()
				oLinkDef = enumLink.Current
				If Not (oLinkDef.Text Is Nothing) AndAlso oLinkDef.Text.IndexOf(strSearchVarName) >= 0 Then
					oLinksNet.Remove(strSearchVarName) ' Avoids duplicate key in COM collection
					oLinksNet.Add(oVarName, KfxLinkSourceType.KFX_REL_VARIABLE, strSearchVarName)
				End If
			End While
		Next
		enumLink = Nothing
		' Iterates entire the document Index Fields and stores field names to the links object for any assigned
		Using enumerator As New ComEnumerator(Me.m_oSetup.SetupDataNet.IIndexFieldsNet.GetEnumerator())
			While enumerator.MoveNext()
				Dim oIndexField As IIndexField = CType(enumerator.Current, IIndexField)
				Using New ComDisposer(oIndexField)
					Dim strSearchIndexName As String = MacroBuilder.BuildMacroName(oIndexField.Name, KfxLinkSourceType.KFX_REL_INDEXFIELD)
					enumLink = linkCollection.GetEnumerator()
					' Iterates entire that collection and add any used document index field
					While enumLink.MoveNext()
						oLinkDef = enumLink.Current
						If Not (oLinkDef.Text Is Nothing) AndAlso oLinkDef.Text.IndexOf(strSearchIndexName) >= 0 Then
							oLinksNet.Remove(strSearchIndexName) ' Avoids duplicate key in COM collection
							oLinksNet.Add(oIndexField.Name, KfxLinkSourceType.KFX_REL_INDEXFIELD, strSearchIndexName)
						End If
					End While
				End Using
			End While
		End Using
		' Iterates entire the batch fields and stores batch field names to the links object for any assinged.
		Using enumerator As New ComEnumerator(Me.m_oSetup.SetupDataNet.IBatchFieldsNet.GetEnumerator())
			While enumerator.MoveNext()
				Dim oIndexBField As IBatchField = CType(enumerator.Current, IBatchField)
				Using New ComDisposer(oIndexBField)
					Dim strSearchBatchName As String = MacroBuilder.BuildMacroName(oIndexBField.Name, KfxLinkSourceType.KFX_REL_BATCHFIELD)
					enumLink = linkCollection.GetEnumerator()
					' Iterates entire that collection and add any used batch fields
					While enumLink.MoveNext()
						oLinkDef = enumLink.Current
						If Not (oLinkDef.Text Is Nothing) AndAlso oLinkDef.Text.IndexOf(strSearchBatchName) >= 0 Then
							oLinksNet.Remove(strSearchBatchName) ' Avoids duplicate key in COM collection
							oLinksNet.Add(oIndexBField.Name, KfxLinkSourceType.KFX_REL_BATCHFIELD, strSearchBatchName)
						End If
					End While
				End Using
			End While
		End Using
	End Sub

	''' <summary>
	''' Loads setup data from KC database for fax settings tab page.
	''' </summary>
	''' <remarks></remarks>
	Private Sub LoadTAPLinsRow()
		Dim ACLink As ILink
		For Each ACLink In m_oSetup.SetupDataNet.ILinksNet
			For Each _row As TAPLinksRow In ctlLnkIndexFields.Rows
				If (_row.Destination.ID.Equals(ACLink.Destination)) Then
					_row.Source.SourceType = ACLink.SourceType
					_row.SourceText = ACLink.Source
					Exit For
				End If
			Next
		Next
	End Sub

	''' <summary>
	''' Renders the TAPLinkRows control based on the fax server type.
    ''' </summary>
    ''' <remarks>
    ''' MB 2011-01-25 ADD THESE 5 FIELDS FOR MAPPING
    ''' -	billing info 1
    ''' -	billing info 2
    ''' -	password
    ''' -	secure doc (this will be true/false)
    ''' -	certified doc (also true/false)
    '''</remarks>
    Private Sub ConstructTAPLinkRows(Optional ByVal bLoadFromKCDatabse As Boolean = False)
        Dim oConnector As IConnector = GetCurrentConnectorFromComboBox()
        If (oConnector Is Nothing) Then Return

        ctlLnkIndexFields.Init(m_oSetup.SetupDataNet)
        ctlLnkIndexFields.RemoveAllRows()
        Dim oDestination As IDestination = New Destination()

        With oDestination
            .DisplayName = Resources.LBL_TAP_FAX_NUMBER_FIELD
            .ID = FaxDestination.c_IdFaxNumberField
            .Required = True
        End With

        ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

        oDestination = New Destination()

        With oDestination
            .DisplayName = Resources.LBL_TAP_FAX_FROM_FIELD
            .ID = FaxDestination.c_IdFaxFromField
            If oConnector.InternalName.Equals(RightFaxConnector.NAME) Then
                .Required = False
            Else
                .Required = True
            End If

        End With

        ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

        If (oConnector.IsToFieldSupported) Then
            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_FAX_TO_FIELD
                .ID = FaxDestination.c_IdToField
                .Required = False
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)
        End If

        oDestination = New FaxDestination()
        With oDestination
            .DisplayName = Resources.LBL_TAP_FAX_SENDER
            .ID = FaxDestination.c_IdSenderField
            .Required = False
        End With
        ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

        If (oConnector.IsSubjectFieldSupported) Then
            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_FAX_SUBJECT_FIELD
                .ID = FaxDestination.c_IdSubjectField
                .Required = False
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)
        End If

        ' MB-CODE 2011-01-26 BEGIN
        ' Add new destinations for setup form field mapping if iConnector is RightFax
        If oConnector.InternalName.Equals(RightFaxConnector.NAME) Then
            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_OWNER_FIELD
                .ID = FaxDestination.c_IdOwnerField
                .Required = False
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_COVERSHEET_TYPE_FIELD
                .ID = FaxDestination.c_IdCoversheetModeField
                .Required = True
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_BILLING_INFO1_FIELD
                .ID = FaxDestination.c_IdBilling1Field
                .Required = False
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_BILLING_INFO2_FIELD
                .ID = FaxDestination.c_IdBilling2Field
                .Required = False
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_SEC_DOC_PASSWORD
                .ID = FaxDestination.c_IdSecurePasswordField
                .Required = False
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)

            oDestination = New FaxDestination()
            With oDestination
                .DisplayName = Resources.LBL_TAP_DELIVERY_TYPE_FIELD
                .ID = FaxDestination.c_IdDeliveryModeField
                .Required = True
            End With
            ctlLnkIndexFields.AddRow(oDestination, String.Empty, True)
        End If
        ' MB-CODE 2011-01-26 END


        If (bLoadFromKCDatabse) Then
            LoadTAPLinsRow()
        End If
    End Sub
#End Region

#Region "Event handlers"

	''' <summary>
	''' Handles the cboFaxServer combobox SelectedIndexChanged event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub cboFaxServer_SelectedIndexChanged(ByVal sender As Object, ByVal e As EventArgs) Handles cboFaxServer.SelectedIndexChanged
		If (m_oSetup.ClsFaxServerSetup.Connector Is Nothing) Then
			Return
		End If
		Dim oConnector As IConnector = GetCurrentConnectorFromComboBox()
		If Not (m_oSetup.ClsFaxServerSetup.Connector.InternalName.Equals(oConnector.InternalName)) Then
			With m_oSetup.ClsFaxServerSetup
				MessageBox.Show(Resources.MSG_WARNING_FAX_SERVER_CHANGE, _
						Resources.COMMON_SHORT_TITLE, _
						MessageBoxButtons.OK, MessageBoxIcon.Information)
			End With
			m_oSetup.ClsFaxServerSetup.Connector = oConnector
			If (TypeOf oConnector Is KofaxCommunicationServerConnector Or _
			 TypeOf oConnector Is RightFaxConnector) Then
				lblPath.Text = Resources.LBL_TXT_PATH
			Else
				lblPath.Text = Resources.LBL_TXT_PATH_BISCOM
			End If
			txtUsername.Text = oConnector.Username
			txtPassword.Text = oConnector.Password
			txtPath.Text = oConnector.Path
			ConstructTAPLinkRows()
			If m_bFormShown Then
				IsDirty = True
			End If
		End If
	End Sub

	''' <summary>
	''' Handles the cboFaxServer combobox SelectionChangeCommitted event.<br/>
	''' When it commits its change, set the saved informations from the current connector to form.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks>This event will happened before SelectedIndexChanged event.</remarks>
	Private Sub cboFaxServer_SelectionChangeCommitted(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles cboFaxServer.SelectionChangeCommitted
		Dim oConnector As IConnector = GetCurrentConnectorFromComboBox()
		txtPath.Text = oConnector.Path
		txtUsername.Text = oConnector.Username
		txtPassword.Text = oConnector.Password
	End Sub

	''' <summary>
	''' Handles the cboFaxServer Enter event.
	''' Saves all entered informations to the current connector.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub cboFaxServer_Enter(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles cboFaxServer.Enter
		Dim oConnector As IConnector = GetCurrentConnectorFromComboBox()
		With oConnector
			.Path = txtPath.Text.Trim()
			.Username = txtUsername.Text.Trim()
			.Password = txtPassword.Text.Trim()
		End With
	End Sub
	''' <summary>
	''' Handles the key down event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks>Switches tab page when the cursor outside the tab control and user press Ctrl + Tab keys.</remarks>
	Private Sub SetupForm_KeyDown(ByVal sender As Object, ByVal e As System.Windows.Forms.KeyEventArgs) Handles Me.KeyDown
		Dim shift As Short = CShort(e.KeyData \ &H10000)
		Dim oForm As Form = CType(sender, Form)
		Dim keyCode As Keys = e.KeyCode
		If Not (TypeOf oForm.ActiveControl Is TabControl) Then
			Dim nTabCount As Integer = ctlTabSetup.TabCount
			Dim nextTab As Integer = 0
			If keyCode = Keys.Tab Then
				If shift = ShiftConstants.CtrlMask Then
					' Calculates the next selected tab. Roll to the first tab if the current tab reaches the total tab page count.
					nextTab = ctlTabSetup.SelectedIndex() + 1
					If (nextTab = nTabCount) Then
						nextTab = 0
					End If
				ElseIf shift = ShiftConstants.CtrlMask + ShiftConstants.ShiftMask Then
					nextTab = ctlTabSetup.SelectedIndex() - 1
					If (nextTab < 0) Then
						' Wraps to the last tab if the first tab is reached and the user press Ctrl + Shift + Tab
						nextTab = nTabCount - 1
					End If
				End If
				ctlTabSetup.SelectTab(nextTab)
			End If
		End If
	End Sub

	''' <summary>
	''' Handles the frmSetup form Shown event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub SetupForm_Shown(ByVal sender As Object, ByVal e As EventArgs) Handles MyBase.Shown
		lblBatchClassName.Text = m_oSetup.BatchClassName
		lblDocumentClassName.Text = m_oSetup.DocumentClassName
		txtName.Text = m_oSetup.ExportConnectorName

		Dim lIndex As Integer
		cboFaxServer.Items.Clear()
		Dim oConnectors As IList(Of IConnector) = ConnectorFactory.GetConnectors()

		' Adding fax server name to combobox
		For Each oConnector As IConnector In oConnectors
			' For loading dll resources
			oConnector.InitializeFax()
			lIndex = cboFaxServer.Items.Add(oConnector)
			If (oConnector.InternalName().Equals(m_oSetup.ClsFaxServerSetup.InternalConnectorName)) Then
				cboFaxServer.SelectedIndex = lIndex
				m_oSetup.ClsFaxServerSetup.Connector = oConnector
			End If
		Next

		' Selects first item if nothing has not been previously selected
		If (-1 = cboFaxServer.SelectedIndex And cboFaxServer.Items.Count > 0) Then
			cboFaxServer.SelectedIndex = 0
			m_oSetup.ClsFaxServerSetup.Connector = CType(cboFaxServer.SelectedItem, IConnector)
		End If

		' Sets the label text based on fax server type
		With m_oSetup.ClsFaxServerSetup
			If (TypeOf .Connector Is KofaxCommunicationServerConnector Or TypeOf .Connector Is RightFaxConnector) Then
				lblPath.Text = Resources.LBL_TXT_PATH
			Else
				lblPath.Text = Resources.LBL_TXT_PATH_BISCOM
			End If
			txtUsername.Text = .Username
			txtPassword.Text = .Password
			txtPath.Text = .Path
		End With

		ctlLnkIndexFields.SourceHeadingText = Resources.LBL_TAP_LNK_SRC
		ctlLnkIndexFields.DestinationHeadingText = Resources.LBL_TAP_LNK_DST
		ConstructTAPLinkRows(True)
		' Fixed SPR00049029 - KCEC-Fax: Setup does not auto focus on deleted Batch/Index fields
		If Not (m_DeletedDestinationID Is Nothing) Then
			Dim control As Control = ctlLnkIndexFields.GetRowByID(m_DeletedDestinationID)
			If Not (control Is Nothing) Then
				ActiveControl = control
				control.Focus()
			End If
		End If
		m_bFormShown = True
	End Sub
	''' <summary>
	''' Handles the frmSetup form FormClosing event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub SetupForm_FormClosing(ByVal sender As Object, ByVal e As FormClosingEventArgs) Handles MyBase.FormClosing
		Try
			Dim bResult As Boolean = Not ConfirmChanges()
			e.Cancel = bResult
			If Not (bResult) Then
				Dim en As IEnumerator
				en = cboFaxServer.Items.GetEnumerator()
				While en.MoveNext()
					Dim oConnector As IConnector = CType(en.Current, IConnector)
					oConnector.DeInitializeFax()
				End While
			End If
		Catch ex As FacsimileException
			MessageBox.Show(ex.Message, _
				 Resources.COMMON_SHORT_TITLE, _
				 MessageBoxButtons.OK, _
				 MessageBoxIcon.Exclamation)
		Catch ex As Exception
			Dim frmUnexpectedException As New UnexpectedExceptionForm(ex)
			frmUnexpectedException.ShowDialog()
		End Try
	End Sub
	''' <summary>
	''' Handles the F1 key press event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="hlpevent"></param>
	''' <remarks></remarks>
	Private Sub SetupForm_HelpRequested(ByVal sender As System.Object, ByVal hlpevent As System.Windows.Forms.HelpEventArgs) Handles MyBase.HelpRequested
		cmdHelp_Click(cmdHelp, New System.EventArgs())
	End Sub
	''' <summary>
	''' Handles the cmdOK button Click event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub cmdOK_Click(ByVal sender As Object, ByVal e As EventArgs) Handles cmdOK.Click
		Try
			If IsDirty Then
				If (SaveSetupData()) Then
					Me.Close()
				End If
			ElseIf m_bFieldsDirty Then
				If (VerifyIndexFieldsUsage()) Then
					Me.Close()
				End If
			Else
				Me.Close()
			End If
		Catch ex As Exception
			Dim frmUnexpectedException As New UnexpectedExceptionForm(ex)
			frmUnexpectedException.ShowDialog()
		End Try
	End Sub

	''' <summary>
	''' Handles the txtPassword textbox TextChanged event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtPassword_TextChanged(ByVal sender As Object, ByVal e As EventArgs) Handles txtPassword.TextChanged
		If m_bFormShown Then
			IsDirty = True
		End If
	End Sub

	''' <summary>
	''' Handles the txtName textbox TextChanged event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtName_TextChanged(ByVal sender As Object, ByVal e As EventArgs) Handles txtName.TextChanged
		If m_bFormShown Then
			IsDirty = True
		End If
	End Sub

	''' <summary>
	''' Handles the txtName textbox Leave event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtName_Leave(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles txtName.Leave
		txtName.Text = txtName.Text.Trim()
	End Sub

	''' <summary>
	''' Handles the txtUsername textbox TextChanged event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtUsername_TextChanged(ByVal sender As Object, ByVal e As EventArgs) Handles txtUsername.TextChanged
		If (String.IsNullOrEmpty(txtPath.Text.Trim()) Or String.IsNullOrEmpty(txtUsername.Text.Trim())) Then
			cmdTestConnection.Enabled = False
		Else
			cmdTestConnection.Enabled = True
		End If
		If m_bFormShown Then
			IsDirty = True
		End If
	End Sub

	''' <summary>
	''' Handles the txtUsername textbox Leave event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtUsername_Leave(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles txtUsername.Leave
		txtUsername.Text = txtUsername.Text.Trim()
	End Sub

	''' <summary>
	''' Handles the txtFaxServerName textbox TextChanged event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtFaxServerName_TextChanged(ByVal sender As Object, ByVal e As EventArgs)
		If m_bFormShown Then
			IsDirty = True
		End If
	End Sub

	''' <summary>
	''' Handles the cmdApply button Click event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub cmdApply_Click(ByVal sender As Object, ByVal e As EventArgs) Handles cmdApply.Click
		Try
			If (SaveSetupData()) Then
				' Disable the Apply button
				IsDirty = False
				' Resets the m_bFieldsDirty, the fields usages validation message has been shown
				If (m_bFieldsDirty) Then
					m_bFieldsDirty = False
				End If
			End If
		Catch ex As Exception
			Dim frmUnexpectedException As New UnexpectedExceptionForm(ex)
			frmUnexpectedException.ShowDialog()
		End Try
	End Sub

	''' <summary>
	''' Handles the cmdTestConnection button Click event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub cmdTestConnection_Click(ByVal sender As Object, ByVal e As EventArgs) Handles cmdTestConnection.Click
		Dim oConnector As IConnector = GetCurrentConnectorFromComboBox()
		Dim strMessage As String = String.Empty
		Try
			If Not (oConnector Is Nothing) Then
				With oConnector
					m_oSetup.ClsFaxServerSetup.Verify()
					.Path = Me.txtPath.Text.Trim()
					.Username = Me.txtUsername.Text.Trim()
					.Password = Me.txtPassword.Text
					.StartFax()
				End With
				' SPR00048946: Verifies that the appropriate client software is installed.
				oConnector.VerifyClientInstalled()
				' Tries to connect to the fax server
				Dim frmWaitForm As WaitingForm = New WaitingForm(oConnector)
				If (frmWaitForm.ShowDialog() = Windows.Forms.DialogResult.OK) Then
					' If the user clicked Stop, then no message is displayed
					strMessage = String.Format(Resources.MSG_INFORMATION_CONNECTION_SUCCESS, oConnector.ToString())
					MessageBox.Show(strMessage, _
						Resources.COMMON_SHORT_TITLE, MessageBoxButtons.OK, MessageBoxIcon.Information)
				ElseIf Not frmWaitForm.IsForceStop And Not (frmWaitForm.Exception Is Nothing) Then
					Throw frmWaitForm.Exception
				End If
			End If
		Catch ex As FacsimileException
			strMessage = ex.Message
			Dim errorCode As Integer = ex.ErrorCode
			If TypeOf ex Is RequireClientMissingException Then
				strMessage = Resources.MSG_ERROR_CLIENT_MISSING_PROBLEM
			Else
				If ("KOFAX_COMMUNICATION_SERVER_FAX".Equals(oConnector.InternalName)) Then
					Select Case errorCode
						' Wrong user name
						Case 586157317
							strMessage = Resources.MSG_ERROR_INVALID_USERNAME
							' Wrong password					
						Case 586157318
							strMessage = Resources.MSG_ERROR_INVALID_PASSWORD
							' Cannot connect successfully. Please check the following: the network is available, 
							'the host name or IP address is valid, and the fax service and fax server are running.
						Case 558891526
							strMessage = Resources.MSG_ERROR_NETWORK_HOST_OR_SERVICE_PROBLEM
						Case Else
							strMessage = Resources.MSG_ERROR_SERVICE_PROBLEM
					End Select
				ElseIf ("CAPTARIS_RIGHTFAX_FAX".Equals(oConnector.InternalName)) Then
					Select Case errorCode
						' Correct information (server or server share, username, password) but network is unplugged
						' Wrong user name/password
						Case 586163469
							strMessage = Resources.MSG_ERROR_NETWORK_OR_CREDENTIAL_PROBLEM
							' Wrong server host/IP
						Case 537928973
							strMessage = Resources.MSG_ERROR_INVALID_HOST_IP
						Case Else
							strMessage = String.Format(Resources.MSG_ERROR_CONNECT_UNEXPECTED_ERROR, strMessage)
					End Select
				ElseIf ("BISCOM_FAXCOM_FAX".Equals(oConnector.InternalName)) Then
					'MessageBox.Show(strMessage, "JUST DEBUG ONLY - Remove me!!!", MessageBoxButtons.OK, MessageBoxIcon.Exclamation)
					Select Case errorCode
						' Correct information (server or server share, username, password) but network is unplugged
						Case 586163982
							strMessage = Resources.MSG_ERROR_NETWORK_HOST_CREDS_OR_SERVER_PROBLEM
							' Wrong server host/IP
							' Wrong user name
							' Wrong password
							' Valid server host/IP, user name, password but fax service is stopped processing
							' Valid server host/IP, user name, password but server is shut down
						Case Else
							strMessage = String.Format(Resources.MSG_ERROR_CONNECT_UNEXPECTED_ERROR, strMessage)
					End Select
				End If
			End If
			MessageBox.Show(strMessage, Resources.COMMON_SHORT_TITLE, MessageBoxButtons.OK, _
				  MessageBoxIcon.Error)
		Catch ex As Exception
			UnexpectedExceptionForm.HandleUnexpectedException(ex)
		Finally
			If Not (oConnector Is Nothing) And oConnector.IsFaxStarted Then
				Try
					oConnector.ShutdownFax()
				Catch shutdownEx As Exception
				End Try
			End If
		End Try
	End Sub

	''' <summary>
	''' Handles the ctlLnkIndexFields TAPLink Control RowChanged event.
	''' </summary>
	''' <param name="row"></param>
	''' <remarks></remarks>
	Private Sub ctlLnkIndexFields_RowChanged(ByVal row As TAPLinksRow) Handles ctlLnkIndexFields.RowChanged
		If m_bFormShown Then
			IsDirty = True
		End If
	End Sub

	''' <summary>
	''' Handles the ctlLnkIndexFields TAPLink TextChanged event
	''' </summary>
	''' <param name="row"></param>
	''' <remarks></remarks>
	Private Sub ctlLnkIndexFields_RowTextChanged(ByVal row As Kofax.TAPCommon.TAPControls.TAPLinksRow) Handles ctlLnkIndexFields.RowTextChanged
		If m_bFormShown Then
			IsDirty = True
		End If
	End Sub

	''' <summary>
	''' Shows the help topic associated with the current tab page when user clicks on the help button
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub cmdHelp_Click(ByVal sender As Object, ByVal e As EventArgs) Handles cmdHelp.Click
        ' Displays the help.        

        If ctlTabSetup.SelectedIndex = Constants.TabPageIndexEnum.FaxServerTab Then
            System.Diagnostics.Process.Start(GeneralUtil.GetHelpFilePath(Constants.c_strHelpFileFaxServerTab))
            'Help.ShowHelp(Me, m_strHelpPath, HelpNavigator.Topic, Constants.c_strHelpTopicFaxServerTab)
        ElseIf ctlTabSetup.SelectedIndex = Constants.TabPageIndexEnum.FaxSettingsTab Then
            System.Diagnostics.Process.Start(GeneralUtil.GetHelpFilePath(Constants.c_strHelpFileFaxSettingTab))
            'Help.ShowHelp(Me, m_strHelpPath, HelpNavigator.Topic, Constants.c_strHelpTopicSettingsTab)
        Else
            System.Diagnostics.Process.Start(GeneralUtil.GetHelpFilePath(Constants.c_strHelpFileDefault))
            'Help.ShowHelp(Me, m_strHelpPath)
        End If
	End Sub

	''' <summary>
	''' Shows the help topic associated with the Fax Server tab page.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="hlpevent"></param>
	''' <remarks></remarks>
	Private Sub ctlTbPgFaxServer_HelpRequested(ByVal sender As System.Object, ByVal hlpevent As System.Windows.Forms.HelpEventArgs) Handles ctlTbPgFaxServer.HelpRequested
		Help.ShowHelp(Me, m_strHelpPath, HelpNavigator.Topic, Constants.c_strHelpTopicFaxServerTab)
	End Sub

	''' <summary>
	''' Shows the help topic associated with the Fax Properties tab page.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="hlpevent"></param>
	''' <remarks></remarks>
	Private Sub ctlTbPgFields_HelpRequested(ByVal sender As System.Object, ByVal hlpevent As System.Windows.Forms.HelpEventArgs) Handles ctlTbPgFields.HelpRequested
		Help.ShowHelp(Me, m_strHelpPath, HelpNavigator.Topic, Constants.c_strHelpTopicSettingsTab)
	End Sub

	''' <summary>
	''' Shows the help topic associated with current tab page.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="hlpevent"></param>
	''' <remarks></remarks>
	Private Sub ctlTabSetup_HelpRequested(ByVal sender As System.Object, ByVal hlpevent As System.Windows.Forms.HelpEventArgs) Handles ctlTabSetup.HelpRequested
		If ctlTabSetup.SelectedIndex = Constants.TabPageIndexEnum.FaxServerTab Then
			Help.ShowHelp(Me, m_strHelpPath, HelpNavigator.Topic, Constants.c_strHelpTopicFaxServerTab)
		ElseIf ctlTabSetup.SelectedIndex = Constants.TabPageIndexEnum.FaxSettingsTab Then
			Help.ShowHelp(Me, m_strHelpPath, HelpNavigator.Topic, Constants.c_strHelpTopicSettingsTab)
		End If
	End Sub

	''' <summary>
	''' Handles the txtPath textbox TextChanged event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtPath_TextChanged(ByVal sender As Object, ByVal e As EventArgs) Handles txtPath.TextChanged
		If (String.IsNullOrEmpty(txtPath.Text.Trim()) Or String.IsNullOrEmpty(txtUsername.Text.Trim())) Then
			cmdTestConnection.Enabled = False
		Else
			cmdTestConnection.Enabled = True
		End If
		If m_bFormShown Then
			IsDirty = True
		End If
	End Sub

	''' <summary>
	''' Handles the txtPath textbox Leave event.
	''' </summary>
	''' <param name="sender"></param>
	''' <param name="e"></param>
	''' <remarks></remarks>
	Private Sub txtPath_Leave(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles txtPath.Leave
		txtPath.Text = txtPath.Text.Trim()
	End Sub
#End Region
End Class