/****************************************************************************
*	(c) Copyright 2009 Kofax, Inc.. All rights reserved.
*	Unauthorized use, duplication or distribution is strictly prohibited.
*****************************************************************************
*
*	File:		FaxInterface.cpp
*
*	Purpose:	CFaxInterface provides COM level access to funtionalities 
*				of tna_fax library.
*
****************************************************************************/

#include "stdafx.h"
#include "FaxInterface.h"

// CFaxInterface
// Defines messages for returning to client.
// 
//IDS_FAX_ALREADY_INITIALIZED     101	Fax interface has already been initialized.
//IDS_FAX_ALREADY_STARTED         102	Fax Interface has already been started.
//IDS_FAX_DLL_LOAD_FAILED         103	Failed to load library tna_fax.dll.
//IDS_FAX_DLL_NOT_COMPATIBLE      104	Application was build with api version %ld which is not compatible with version %ld
//IDS_FAX_GET_INTERFACE_FAILED    105	Failed to get Fax Interface.
//IDS_FAX_NOT_INITIALIZED         106	Fax Interface has not been initialized.
//IDS_FAX_NOT_STARTED             107	Fax Interface has not been started.
//IDS_FAX_OPERATION_FAILED        108	Failed
//IDS_FAX_OPERATION_OK            109	OK
//IDS_FAX_XSLT_CONFIG_NOT_FOUND   110	The XSLT configuration file could not be found.


// Defines error codes for the client
#define E_FAX_OPERATION_OK			0L
#define E_FAX_NOT_INITIALIZED		1L
#define E_FAX_ALREADY_INITIALIZED	2L
#define E_DLL_LOAD_FAILED			3L
#define E_GET_INTERFACE_FAILED		4L
#define E_FAX_NOT_STARTED			5L
#define E_FAX_ALREADY_STARTED		6L
#define E_XSLT_CONFIG_NOT_FOUND		7L
#define E_DLL_NOT_COMPATIBLE		8L
// Implements of the ISupportsErrorInfo interface
STDMETHODIMP CFaxInterface::InterfaceSupportsErrorInfo(REFIID riid)
{
	static const IID* arr[] = 
	{
		&IID_IFaxInterface
	};

	for (int i=0; i < sizeof(arr) / sizeof(arr[0]); i++)
	{
		if (InlineIsEqualGUID(*arr[i],riid))
			return S_OK;
	}
	return S_FALSE;
}



// Purpose: Starts the C++ fax interface.
// param: bstrAppPath [in] the directory where the program started
// param: bstrRawConfig [in] the raw configuration in XML format.
// param: bstrXSLTPath [in] the XSLT file path for transformation inside C++ interface.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
STDMETHODIMP CFaxInterface::Start(BSTR bstrAppPath, 
								  BSTR bstrRawConfig, 
								  BSTR bstrXSLTPath)
{
	ATLTRACE(_T("StartFax is called."));
	WriteEventLog("StartFax is called.", EVENTLOG_INFORMATION_TYPE);
	if (!m_bInitialised)
	{
		return ReportError(E_FAX_NOT_INITIALIZED, IDS_FAX_NOT_INITIALIZED);
	}
	if (m_hModuleDLL == 0)
	{
		return ReportError(E_DLL_LOAD_FAILED, IDS_FAX_DLL_LOAD_FAILED);
	}
	if (m_pFax == 0)
	{
		return ReportError(E_GET_INTERFACE_FAILED, IDS_FAX_GET_INTERFACE_FAILED);
	}
	if ( m_bInterfaceStarted ) 
	{
		m_pFax->Shutdown();
	}	
	LPCTSTR pszAppPath = COLE2CT(bstrAppPath);
	LPCTSTR pszRawConfig = COLE2CT(bstrRawConfig);
	if (bstrXSLTPath != NULL)
	{
		BOOL xsltExists = IsFileExist(bstrXSLTPath);
		if (!xsltExists) 
		{
			return ReportError(E_XSLT_CONFIG_NOT_FOUND, IDS_FAX_XSLT_CONFIG_NOT_FOUND);
		}
	}
	kofax::tna::ResultPtr res_ptr;
	if ( bstrXSLTPath != NULL)
	{
		LPCTSTR pszXSLTPath = COLE2CT(bstrXSLTPath);
		res_ptr = m_pFax->Start(CW2A(pszAppPath), CW2A(pszRawConfig), CW2A(pszXSLTPath));
	}
	else
	{
		res_ptr = m_pFax->Start(CW2A(pszAppPath), CW2A(pszRawConfig));
	}
	if ( !res_ptr->IsOK() )
	{
		WriteEventLog(res_ptr->GetErrorDescription(), EVENTLOG_ERROR_TYPE);
		return ReportError(res_ptr->GetErrorCode(), res_ptr->GetErrorDescription());
	}
	m_bInterfaceStarted = TRUE;
	WriteEventLog("StartFax is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}

// Purpose: Shuts down the fax interface. Tells the fax interface to free any resources used.
// param: nTimeout [in] the shut down time-out.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
//			E_POINTER, E_NOINTERFACE if out of memory or no interface.
STDMETHODIMP CFaxInterface::Shutdown(LONG lTimeout)
{
	ATLTRACE(_T("Shutdown is called."));
	WriteEventLog("Shutdown is called.", EVENTLOG_INFORMATION_TYPE);
	if (!m_bInitialised)
	{
		return ReportError(E_FAX_NOT_INITIALIZED, IDS_FAX_NOT_INITIALIZED);
	}
	if (m_hModuleDLL == 0)
	{
		return ReportError(E_DLL_LOAD_FAILED, IDS_FAX_DLL_LOAD_FAILED);
	}
	if (m_pFax == 0)
	{
		return ReportError(E_GET_INTERFACE_FAILED, IDS_FAX_GET_INTERFACE_FAILED);
	}
	if (!m_bInterfaceStarted)
	{
		return ReportError(E_FAX_NOT_STARTED, IDS_FAX_NOT_STARTED);
	}
	HRESULT hr = S_FALSE;
	int nTimeOut = (int)lTimeout;
	if ( nTimeOut <= 10000 )
		nTimeOut = 10000;
	kofax::tna::ResultPtr res_ptr = m_pFax->Shutdown(nTimeOut);
	m_bInterfaceStarted = false;
	if ( !res_ptr->IsOK() )
	{
		WriteEventLog(res_ptr->GetErrorDescription(), EVENTLOG_ERROR_TYPE);
		return ReportError(res_ptr->GetErrorCode(), res_ptr->GetErrorDescription());
	}
	WriteEventLog("Shutdown is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}

// Purpose: Gets the active server properties.
// param: pResponse [in,out] the server properties.
// param: bstrUser [in] the username used to retrieve the fax server properties.
// param: bstrPassword [in] the user's password used to retrieve the fax server properties.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
//			E_POINTER, E_NOINTERFACE if out of memory or no interface.
STDMETHODIMP CFaxInterface::GetFaxServerProperties(BSTR szUser, 
												BSTR szPassword, 
												IServerProperties** pResponse)
{
	ATLTRACE(_T("GetServerProperties is called."));
	WriteEventLog("GetServerProperties is called.", EVENTLOG_INFORMATION_TYPE);
	if (!m_bInitialised)
	{
		return ReportError(E_FAX_NOT_INITIALIZED, IDS_FAX_NOT_INITIALIZED);
	}
	if (m_hModuleDLL == 0)
	{
		return ReportError(E_DLL_LOAD_FAILED, IDS_FAX_DLL_LOAD_FAILED);
	}
	if (m_pFax == 0)
	{
		return ReportError(E_GET_INTERFACE_FAILED, IDS_FAX_GET_INTERFACE_FAILED);
	}
	if (!m_bInterfaceStarted)
	{
		return ReportError(E_FAX_NOT_STARTED, IDS_FAX_NOT_STARTED);
	}
	HRESULT hr;
	CComPtr<IServerProperties> pIServerProp;
	hr = pIServerProp.CoCreateInstance(CLSID_ServerProperties);
	if ( FAILED(hr) ) return hr;
	if ( pIServerProp == NULL ) return E_POINTER;
	LPCTSTR pszUser = COLE2CT(szUser);
	LPCTSTR pszPassword = COLE2CT(szPassword);
	kofax::tna::ServerPropertiesPtr response_ptr;
	kofax::tna::ResultPtr result_ptr = m_pFax->GetServerProperties(&response_ptr, CW2A(pszUser), CW2A(pszPassword));
	if ( !result_ptr->IsOK() )
	{
		WriteEventLog(result_ptr->GetErrorDescription(), EVENTLOG_ERROR_TYPE);
		return ReportError(result_ptr->GetErrorCode(), result_ptr->GetErrorDescription());
	}
	pIServerProp->put_FaxServerType(CComBSTR(response_ptr->GetServerType()));
	pIServerProp->put_FaxServerVersion(CComBSTR(response_ptr->GetFaxServerVersion()));
	*pResponse = pIServerProp.Detach();
	WriteEventLog("GetServerProperties is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}

// Purpose: Sends the fax from an existing TIFF file.
// param: bstrUser [in] the username that being used to send the fax.
// param: bstrPassword [in] the user's password that being used to send the fax.
// param: bstrTiffFilePath [in] the path to the TIF file that being sent to fax.
// param: bstrNumberToDial [in] the fax number to dial.
// param: bstrExpectedCSI [in] the expected called subscriber identification, or a part of it, for verification that the correct 
//		  destination has been called. The sending proceeds only if the string specified here can be found somewhere within the received CSI.
// param: bstrNameOfRecipient [in] the recipient's name that being inserted into fax header line.
// param: bstrSourceTSI [in] the origin subscriber identification, the recipient should send a reply to this fax number.
// param: bstrNameOfSender [in] the sender name that being inserted into fax header line if supported.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
STDMETHODIMP CFaxInterface::SendFaxFromFile(BSTR bstrUser, 
											BSTR bstrPassword, 
											BSTR bstrTiffFilePath, 
											BSTR bstrNumberToDial, 
											BSTR bstrExpectedCSI, 
											BSTR bstrNameOfRecipient, 
											BSTR bstrSourceTSI, 
											BSTR bstrNameOfSender, 
											BSTR bstrSubject)
{
	ATLTRACE(_T("SendFaxFromFile is called."));
	WriteEventLog("SendFaxFromFile is called.", EVENTLOG_INFORMATION_TYPE);
	if (!m_bInitialised)
	{		
		return ReportError(E_FAX_NOT_INITIALIZED, IDS_FAX_NOT_INITIALIZED);
	}
	if (m_hModuleDLL == 0)
	{
		return ReportError(E_DLL_LOAD_FAILED, IDS_FAX_DLL_LOAD_FAILED);
	}
	if (m_pFax == 0)
	{
		return ReportError(E_GET_INTERFACE_FAILED, IDS_FAX_GET_INTERFACE_FAILED);
	}
	if (!m_bInterfaceStarted)
	{
		return ReportError(E_FAX_NOT_STARTED, IDS_FAX_NOT_STARTED);
	}
	HRESULT hr = S_FALSE;
	LPCTSTR pszUser = COLE2CT(bstrUser);
	LPCTSTR pszPassword = COLE2CT(bstrPassword);
	LPCTSTR pszTiffFile = COLE2CT(bstrTiffFilePath);
	LPCTSTR pszNumberToDial = COLE2CT(bstrNumberToDial);
	LPCTSTR pszExpectedCSI = COLE2CT(bstrExpectedCSI);
	LPCTSTR pszNameOfRecipient = COLE2CT(bstrNameOfRecipient);
	LPCTSTR pszSourceTSI = COLE2CT(bstrSourceTSI);
	LPCTSTR pszNameOfSender = COLE2CT(bstrNameOfSender);
	LPCTSTR pszSubject = COLE2CT(bstrSubject);
	// Sends the fax from file
	// SPR 96455 - Call SendFaxFromFile2 as a workaround
	// Note, we call SendFaxFromFile2 as there is an infinite recursion bug in the current version of tna_fax::SendFaxFromFile.  The
	// intention was for SendFaxFromFile to chain to SendFaxFromFile2 internally, however the 2 was missing from the call.
	// As a workaround, call SendFaxFromFile2 ourselves.
	kofax::tna::ResultPtr result_ptr = m_pFax->SendFaxFromFile2(CW2A(pszUser), CW2A(pszPassword), CW2A(pszTiffFile), 
		CW2A(pszNumberToDial), CW2A(pszExpectedCSI), CW2A(pszNameOfRecipient), CW2A(pszSourceTSI), 
		CW2A(pszNameOfSender), CW2A(pszSubject));
	if ( !result_ptr->IsOK() )
	{
		WriteEventLog(result_ptr->GetErrorDescription(), EVENTLOG_ERROR_TYPE);
		return ReportError(result_ptr->GetErrorCode(), result_ptr->GetErrorDescription());
	}
	WriteEventLog("SendFaxFromFile is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}


// Purpose: Sends the fax from an existing TIFF file with new RightFax functionality extentions.
// param: bstrUser [in] the username that being used to send the fax.
// param: bstrPassword [in] the user's password that being used to send the fax.
// param: bstrTiffFilePath [in] the path to the TIF file that being sent to fax.
// param: bstrNumberToDial [in] the fax number to dial.
// param: bstrExpectedCSI [in] the expected called subscriber identification, or a part of it, for verification that the correct 
//		  destination has been called. The sending proceeds only if the string specified here can be found somewhere within the received CSI.
// param: bstrNameOfRecipient [in] the recipient's name that being inserted into fax header line.
// param: bstrSourceTSI [in] the origin subscriber identification, the recipient should send a reply to this fax number.
// param: bstrCoverMode [in] 
//			0 sends the document w/o cover sheet
//			1 (=default) uses the default server specific behavior
//			2 sends the document with cover sheet
// param: bstrOwner [in] User name to use in sending Fax.
// param: bstrBilling1 [in] Billing 1 (Account) name.
// param: bstrBilling2 [in] Billing 2 (Matter) name.
// param: bstrDeliveryMode [in] 
//			0 (DM_FAX, default)  sends the messages as Fax 
//			1 (DM_SECURE_DOC) sends a secure document
//			2 (DM_CERTIFIED_DELIVERY) sends a secure document with certified delivery.
// param: bstrSecureDeliveryPassword [in] Password if secure delivery mode is chosen.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
// MB-CODE 2011-01-26 BEGIN
STDMETHODIMP CFaxInterface::SendFaxFromFileExt(		
		BSTR bstrUser, 
		BSTR bstrPassword, 
		BSTR bstrTiffFilePath, 
		BSTR bstrNumberToDial, 
		BSTR bstrExpectedCSI, 
		BSTR bstrNameOfRecipient, 
		BSTR bstrSourceTSI, 
		BSTR bstrNameOfSender, 
		BSTR bstrSubject, 
		BSTR bstrCoverMode,
		BSTR bstrOwner,
		BSTR bstrBilling1,
		BSTR bstrBilling2,
		BSTR bstrDeliveryMode,
		BSTR bstrSecureDeliveryPassword)

{
	ATLTRACE(_T("SendFaxFromFileExt is called."));
	WriteEventLog("SendFaxFromFileExt is called.", EVENTLOG_INFORMATION_TYPE);	
	if (!m_bInitialised)
	{
		return ReportError(E_FAX_NOT_INITIALIZED, IDS_FAX_NOT_INITIALIZED);
	}
	if (m_hModuleDLL == 0)
	{
		return ReportError(E_DLL_LOAD_FAILED, IDS_FAX_DLL_LOAD_FAILED);
	}
	if (m_pFax == 0)
	{
		return ReportError(E_GET_INTERFACE_FAILED, IDS_FAX_GET_INTERFACE_FAILED);
	}
	if (!m_bInterfaceStarted)
	{
		return ReportError(E_FAX_NOT_STARTED, IDS_FAX_NOT_STARTED);
	}
	HRESULT hr = S_FALSE;
	LPCTSTR pszUser = COLE2CT(bstrUser);
	LPCTSTR pszPassword = COLE2CT(bstrPassword);
	LPCTSTR pszTiffFile = COLE2CT(bstrTiffFilePath);
	LPCTSTR pszNumberToDial = COLE2CT(bstrNumberToDial);
	LPCTSTR pszExpectedCSI = COLE2CT(bstrExpectedCSI);
	LPCTSTR pszNameOfRecipient = COLE2CT(bstrNameOfRecipient);
	LPCTSTR pszSourceTSI = COLE2CT(bstrSourceTSI);
	LPCTSTR pszNameOfSender = COLE2CT(bstrNameOfSender);
	LPCTSTR pszSubject = COLE2CT(bstrSubject);

	// new fields added for this interface:
	LPCTSTR pszOwner = COLE2CT(bstrOwner);
	LPCTSTR pszBilling1 = COLE2CT(bstrBilling1);
	LPCTSTR pszBilling2 = COLE2CT(bstrBilling2);
	LPCTSTR pszSecureDeliveryPassword = COLE2CT(bstrSecureDeliveryPassword);
	
	// convert to int for next conversion to enum
	int intCoverMode = _wtoi(bstrCoverMode);
	int intDeliveryMode = _wtoi(bstrDeliveryMode);

	// Sends the fax from file
	kofax::tna::ResultPtr result_ptr = m_pFax->SendFaxFromFile2(
		CW2A(pszUser), 
		CW2A(pszPassword), 
		CW2A(pszTiffFile), 
		CW2A(pszNumberToDial), 
		CW2A(pszExpectedCSI), 
		CW2A(pszNameOfRecipient), 
		CW2A(pszSourceTSI), 
		CW2A(pszNameOfSender), 
		CW2A(pszSubject),

		// new fields added for this interface:
		static_cast<kofax::tna::FaxInterface::CoverModes>(intCoverMode),
		CW2A(pszOwner),
		CW2A(pszBilling1),
		CW2A(pszBilling2),
		static_cast<kofax::tna::FaxInterface::DeliveryModes>(intDeliveryMode),
		CW2A(pszSecureDeliveryPassword));

	if ( !result_ptr->IsOK() )
	{
		WriteEventLog(result_ptr->GetErrorDescription(), EVENTLOG_ERROR_TYPE);
		return ReportError(result_ptr->GetErrorCode(), result_ptr->GetErrorDescription());
	}
	WriteEventLog("SendFaxFromFileExt is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}
// MB-CODE 2011-01-26 END


// Purpose: Determines if the current version of the wrapper is compatible with
//			the fax library version.
// param: lExpectedVersion [in] the parameter must be set to the API version used during build of the application. 
// param: lMinVersion [in] the You may optionaly specify the used API features using to lowest API which supports
//		  all this features. If you do not know this version use 0.
// param: pCurrentVersion [in, out] the LONG pointer that holds the current TNA_Fax version.
// param: pCompatible [in, out] the Boolean pointer that holds the Boolean value which indicates whether this wrapper is compatible to TNA_Fax library.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
//			E_POINTER, E_NOINTERFACE if out of memory or no interface.
STDMETHODIMP CFaxInterface::IsCompatible(LONG nExpectedVersion, 
										 LONG nMinVersion, 
										 LONG* pCurrentVersion, 
										 VARIANT_BOOL* pCompatible)
{
	ATLTRACE(_T("IsCompatible is called."));
	WriteEventLog("IsCompatible is called.", EVENTLOG_INFORMATION_TYPE);
	if (!m_bInitialised)
	{
		return ReportError(E_FAX_NOT_INITIALIZED, IDS_FAX_NOT_INITIALIZED);
	}
	if (m_hModuleDLL == 0)
	{
		return ReportError(E_DLL_LOAD_FAILED, IDS_FAX_DLL_LOAD_FAILED);
	}
	if (m_pFax == 0)
	{
		return ReportError(E_GET_INTERFACE_FAILED, IDS_FAX_GET_INTERFACE_FAILED);
	}
	if (NULL == pCurrentVersion)
		return E_POINTER;
	if (NULL == pCompatible)
		return E_POINTER;
	int dll_version;
	if ( !m_pFax->IsCompatible(nExpectedVersion, nMinVersion, &dll_version) )
	{
		*pCompatible = VARIANT_FALSE;
	}
	else
	{
		*pCompatible = VARIANT_TRUE;
	}
	*pCurrentVersion = dll_version;
	WriteEventLog("IsCompatible is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}

// Purpose: Initializes the fax interface. Loads the tna_fax.dll into memory 
//			and initialize the fax interface from DLL's handle.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
//			E_POINTER, E_NOINTERFACE if out of memory or no interface.
STDMETHODIMP CFaxInterface::InitializeFax()
{
	ATLTRACE(_T("InitializeFax is called."));
	WriteEventLog("InitializeFax is called.", EVENTLOG_INFORMATION_TYPE);
	if ( m_bInitialised )
	{
		return ReportError(E_FAX_ALREADY_INITIALIZED, IDS_FAX_ALREADY_INITIALIZED);
	}
	m_hModuleDLL = LoadLibrary(L"tna_fax.dll");
	if ( m_hModuleDLL == 0 ) 
	{
		return ReportError(E_DLL_LOAD_FAILED, IDS_FAX_DLL_LOAD_FAILED);
	}
	ATLTRACE(_T("The tna_fax.dll is loaded."));
	LPCSTR lpInterfaceName = kofax::tna::FaxInterface::GetInterfaceName();
	m_pFax = (kofax::tna::FaxInterface*)GetProcAddress(m_hModuleDLL, lpInterfaceName);
	if ( m_pFax == 0 )
	{
		return ReportError(E_GET_INTERFACE_FAILED, IDS_FAX_GET_INTERFACE_FAILED);
	}
	ATLTRACE(_T("Got the fax interface."));
	m_bInitialised = TRUE;
	WriteEventLog("InitializeFax is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}

// Purpose: De-Initializes the fax interface. Un-Loads the tna_fax.dll from 
//			the memory.
// Returns: S_OK if operation performed successful, S_FALSE if operation failed.
STDMETHODIMP CFaxInterface::DeInitializeFax()
{
	ATLTRACE(_T("DeInitializeFax is called."));
	WriteEventLog("DeInitializeFax is called.", EVENTLOG_INFORMATION_TYPE);
	if (m_pFax != NULL)
	{
		if (m_bInterfaceStarted)
			m_pFax->Shutdown();
		m_pFax = 0;
	}
	if (m_hModuleDLL != NULL)
	{
		//SPR 96370 - temporarly disabled since unloading the tna_fax.dll is causing the memory leak
		//FreeLibrary(m_hModuleDLL);
		m_hModuleDLL = NULL;
	}
	// Resets the initialized flag
	m_bInitialised = FALSE;
	WriteEventLog("DeInitializeFax is done.", EVENTLOG_INFORMATION_TYPE);
	return S_OK;
}

// Purpose: Determines if the specified file name is exist or not.
// param: szFileName[in] the file name to check.
// Returns: TRUE if the file is exist and accessible, otherwise FALSE.
BOOL CFaxInterface::IsFileExist(TCHAR* szFileName)
{
	HANDLE hFile = CreateFile(szFileName,			  // file to open
						   GENERIC_READ,          // open for reading
						   FILE_SHARE_READ,       // share for reading
						   NULL,                  // default security
						   OPEN_EXISTING,         // existing file only
						   FILE_ATTRIBUTE_NORMAL, // normal file
						   NULL);                 // no attr. template
	 
	if (hFile == INVALID_HANDLE_VALUE) 
	{ 
		return FALSE; 
	}
	CloseHandle(hFile);
	return TRUE;
}

// Purpose: Declares a globale method to get a string name in the DLL's resource.
// param: uID [in] the resource ID.
// Returns: The string value.
TCHAR* CFaxInterface::LoadStringResource(UINT uID)
{
	TCHAR strBuf[256];
	LoadString(_AtlBaseModule.m_hInst, uID, strBuf, sizeof(strBuf) / sizeof(TCHAR));
	size_t len = wcslen(strBuf) + sizeof(TCHAR);
	TCHAR* strResult = new TCHAR[len];
	wcscpy_s(strResult, len, strBuf);
	return strResult;
}
// Purpose: Reports the error to the caller.
// param: nErrorCode [in] the error code number.
// param: uIDErrorMessage [in] the resource ID of the error message.
// Returns: Error code.
HRESULT CFaxInterface::ReportError(long nErrorCode, UINT uIDErrorMessage)
{
	HRESULT hError = nErrorCode <= 0 ? nErrorCode : -nErrorCode;
	return AtlReportError(CLSID_FaxInterface, uIDErrorMessage, IID_IFaxInterface, hError, _AtlBaseModule.m_hInstResource);
}
// Purpose: Reports the error to the caller.
// param: nErrorCode [in] the error code number.
// param: strErrorMessage [in] the message that describes the error.
// Returns: Error code.
HRESULT CFaxInterface::ReportError(long nErrorCode, LPCSTR strErrorMessage)
{
	HRESULT hError = nErrorCode <= 0 ? nErrorCode : -nErrorCode;
	return AtlReportError(CLSID_FaxInterface, strErrorMessage, 0, NULL, IID_IFaxInterface, hError);
}
bool CFaxInterface::WriteEventLog(LPCSTR szEventMsg, WORD wEventType)
{
	wchar_t unicodeStr[1024];
	MultiByteToWideChar(CP_ACP, 0, szEventMsg, -1, unicodeStr, 1024);
	return WriteEventLog(unicodeStr, wEventType);
}
bool CFaxInterface::WriteEventLog(LPCTSTR szEventMsg, WORD wEventType)
{
	HANDLE hEventLog;
	BOOL bSuccesful;
	bool bReturn = true;
	LPCTSTR szSourceName = L"Tna_Fax COM";
	//-------------------------------------------------------------------------  
	hEventLog = RegisterEventSource(NULL, szSourceName);  
	if(NULL == hEventLog)
	{  
	    // RegisterEventSource Failed
	    return false;
	}  
	bSuccesful = ReportEvent(hEventLog,
                         wEventType,
                         NULL,
                         NULL,
                         NULL,
                         1,
                         0,
                         &szEventMsg,  
                         NULL);  
	if(FALSE == bSuccesful)  
	{
		// ReportEvent Failed
		bReturn = false;
	}
	DeregisterEventSource(hEventLog);  
	return bReturn;  
}